//import { Contact } from '../models/contact.model.js';
import { Contact } from '../models/contact.js';
import { 
    getContactById, 
    getCustomersByContact, 
    getProjectsByContact, 
    getTasksByContact, 
    getFilesByContact
} from '../models/contact.model.js';

export const createContact = async (req, res) => {
    try {
        //console.log(typeof Contact.checkContactExists); // should print 'function'
        // Check if contact already exists
        const exists = await Contact.checkContactExists(
            req.body.email, 
            req.body.mobile_no
        );
        
        if (exists) {
            return res.status(409).json({
                status: 409,
                message: 'Contact with this email or mobile number already exists'
            });
        }

        const contactId = await Contact.create(req.body, req.user.id);
        const contact = await Contact.findById(contactId);
        
        res.status(201).json({
            status: 201,
            message: 'Contact created successfully',
            contact
        });
    } catch (error) {
        console.error('Create contact error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to create contact'
        });
    }
};

export const updateContact = async (req, res) => {
    try {
        const { id } = req.params;

        // Check if contact already exists
        const exists = await Contact.checkContactExists(
            req.body.email, 
            req.body.mobile_no,
            id
        );
        
        if (exists) {
            return res.status(409).json({
                status: 409,
                message: 'Contact with this email or mobile number already exists'
            });
        }

        await Contact.update(id, req.body);
        const contact = await Contact.findById(id);

        res.json({
            status: 200,
            message: 'Contact updated successfully',
            contact
        });
    } catch (error) {
        console.error('Update contact error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to update contact'
        });
    }
};

export const deleteContact = async (req, res) => {
    try {
        const { id } = req.params;
        await Contact.softDelete(id);

        res.json({
            status: 200,
            message: 'Contact deleted successfully'
        });
    } catch (error) {
        console.error('Delete contact error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to delete contact'
        });
    }
};

export const listContacts = async (req, res) => {
    try {
        const contacts = await Contact.list();
        
        res.json({
            status: 200,
            message: 'Contacts retrieved successfully',
            contacts
        });
    } catch (error) {
        console.error('List contacts error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to retrieve contacts'
        });
    }
};

export const listActiveContactsByCompany = async (req, res) => {
    try {
        const { companyId } = req.params;
        if (!companyId) {
            return res.status(400).json({
                status: 400,
                message: 'Company ID is required'
            });
        }

        const contacts = await Contact.listActiveByCompanyId(companyId);

        res.json({
            status: 200,
            message: 'Active contacts retrieved successfully',
            contacts
        });
    } catch (error) {
        console.error('List active contacts by company error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to retrieve active contacts'
        });
    }
};
export const getContactDetailsPage = async (req, res) => {
    const contactId = req.params.id;
    try {
        const contact = await getContactById(contactId);
        const customers = await getCustomersByContact(contactId);
        const projects = await getProjectsByContact(contactId);
        const tasks = await getTasksByContact(contactId);
        const files = await getFilesByContact(contactId);

        res.json({
            status: true,
            message: "Contact details fetched successfully",
            data: {
                contact,
                customers,
                projects,
                tasks,
                files
            }
        });
    } catch (error) {
        res.status(500).json({
            status: false,
            message: "Error fetching contact details",
            error: error.message
        });
    }
};

export const datatablecontacts = async (req, res) => {
    try {
        const {
            start = 1,
            length = 10,
            order_by = 'created_at',
            order_dir = 'DESC',
            search = ''
        } = req.query;

        const page = parseInt(start);
        const limit = parseInt(length);
        const sortBy = order_by;
        const sortOrder = order_dir;

        const { contacts, total } = await Contact.datatableList({ page, limit, sortBy, sortOrder, search });
        const totalPages = Math.ceil(total / limit);
        const paginationMeta = {
            total,
            page,
            limit,
            totalPages
        };
        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Contacts list fetched successfully',
            data: contacts,
            total,
            pagination: paginationMeta
        });
    } catch (error) {
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Error fetching datatable contacts',
            error: error.message
        });
    }
};