import { Conveyance } from '../models/conveyance.js';

export const adminConveyance = async (req, res) => {
    try {
        const { start_date, end_date, branch_id } = req.body;

        if (!start_date || !end_date) {
            return res.status(200).json({
                code: 400,
                status: false,
                message: 'Start date and end date are required',
                response: {}
            });
        }

        const conveyance = await Conveyance.getAdminConveyance({ start_date, end_date, branch_id });

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Conveyance info',
            response: {
                conveyance
            }
        });
    } catch (error) {
        console.error('Admin conveyance error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'Error - ' + error.message,
            response: {}
        });
    }
};

export const getConveyance = async (req, res) => {
    try {
        const { start_date, end_date } = req.query;
        
        // Validate date format if provided
        if (start_date && !isValidDate(start_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid start_date format. Use YYYY-MM-DD"
            });
        }
        
        if (end_date && !isValidDate(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid end_date format. Use YYYY-MM-DD"
            });
        }

        const conveyance = await Conveyance.findAllConveyance(start_date, end_date);
        res.status(200).json(
            {
                status: 200,
                success: true,
                message: "Conveyance list fetched successfully",
                data: conveyance
            }
        );
    } catch (error) {
        res.status(500).json(
            {
                status: 500,
                success: false,
                message: "Error fetching conveyance",
                error: error.message
            }
        );
    }
}

export const  getConveyanceByUserId = async (req, res) => {
    try {
        const { id } = req.params;
        const { start_date, end_date } = req.query;
        
        // Validate user ID
        if (!id || isNaN(id)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid user ID"
            });
        }
        
        // Validate date format if provided
        if (start_date && !isValidDate(start_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid start_date format. Use YYYY-MM-DD"
            });
        }
        
        if (end_date && !isValidDate(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid end_date format. Use YYYY-MM-DD"
            });
        }

const conveyance = await Conveyance.findConveyanceByUserId(id, start_date, end_date);
        res.status(200).json(
            {
                status: 200,
                success: true,
                message: "Conveyance details fetched successfully",
                data: conveyance
            }
        );
    } catch (error) {
        res.status(500).json(
            {
                status: 500,
                success: false,
                message: "Error fetching conveyance by user ID",
                error: error.message
            }
        );
    }
}

export const getClaims = async (req, res) => {
    try {
        const { start_date, end_date } = req.query;
        console.log({start_date, end_date});
        const claims = await Conveyance.findAllClaims(start_date, end_date);
        res.status(200).json(
            {
                status: 200,
                success: true,
                message: "Claims fetched successfully",
                data: claims
            }
        );
    } catch (error) {
        res.status(500).json(
            {
                status: 500,
                success: false,
                message: "Error fetching claims",
                error: error.message
            }
        );
    }
}

export const getFinanceClaims = async (req, res) => {
    try {
        const { start_date, end_date } = req.query;
        console.log({start_date, end_date});

        // Get user ID from authenticated user
        const userId = req.user.id;
        if (!userId) {
            return res.status(200).json({
                status: 400,
                success: false,
                message: "User authentication failed",
                data: []
            });
        }

        // Fetch user details from database to get branch_id
        const userDetails = await Conveyance.getUserById(userId);
        if (!userDetails || !userDetails.branch_id) {
            return res.status(200).json({
                status: 400,
                success: false,
                message: "User branch information not found",
                data: []
            });
        }
         console.error("Error fetching finance claims:", req.user.id);
        // Convert comma-separated branch_id to array
        const userBranchIds = userDetails.branch_id.split(',').map(id => parseInt(id.trim())).filter(id => !isNaN(id));
         console.error("Error fetching finance claims:", userBranchIds);
        const claims = await Conveyance.findFinanceClaims(start_date, end_date, userBranchIds);
        res.status(200).json(
            {
                status: 200,
                success: true,
                message: "Finance approved claims fetched successfully",
                data: claims
            }
        );
    } catch (error) {
        res.status(500).json(
            {
                status: 500,
                success: false,
                message: "Error fetching finance claims",
                error: error.message
            }
        );
    }
}

export const updateClaimStatus = async (req, res) => {
    try {
        const { status, remarks, approved_by } = req.body;
        const { id } = req.params;
        // const approved_by = req.user.id;
        // console.log({approved_by, status, remarks, id });
        if (status === 'Approved') {
        const claim = await Conveyance.updateClaimStatus(id, status, remarks, approved_by);
        res.status(200).json(
            {
                status: 200,
                success: true,
                message: "Claim Approved successfully",
                data: claim
            }
        );
        }else if(status === 'Rejected'){
            const claim = await Conveyance.updateClaimStatus(id, status, remarks, approved_by);
            res.status(200).json(
                {
                    status: 200,
                    success: true,
                    message: "Claim Rejected successfully",
                    data: claim
                }
            );
        }else{
            res.status(400).json(
                {
                    status: 400,
                    success: false,
                    message: "Invalid status"
                }
            );
        }
    } catch (error) {   
        res.status(500).json(
            {
                status: 500,
                success: false,
                message: "Error updating claim status",
                error: error.message
            }
        );
    }
}

export const datatableclaims = async (req, res) => {
    try {
        const {
    start = 1,
    length = 10,
    order_by = 'checkin_at',
    order_dir = 'DESC',
    search = '',
    start_date,
    end_date
} = req.query;

const page = parseInt(start);
const limit = parseInt(length);
const sortBy = order_by;
const sortOrder = order_dir;

const { claims, total } = await Conveyance.datatableClaims({
    page,
    limit,
    sortBy,
    sortOrder,
    search,
    startDate: start_date,
    endDate: end_date
});
        const totalPages = Math.ceil(total / limit);
        const paginationMeta = {
            total,
            page,
            limit,
            totalPages
        };
        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Claims fetched successfully',
            data: claims,
            total,
            pagination: paginationMeta
        });
    } catch (error) {
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Error fetching datatable claims',
            error: error.message
        });
    }
};

export const datatableFinanceClaims = async (req, res) => {
    try {
        const {
            start = 1,
            length = 10,
            order_by = 'checkin_at',
            order_dir = 'DESC',
            search = ''
        } = req.query;

        const page = parseInt(start);
        const limit = parseInt(length);
        const sortBy = order_by;
        const sortOrder = order_dir;

        // Get user ID from authenticated user
        const userId = req.user.id;
        if (!userId) {
            return res.status(200).json({
                status: 400,
                success: false,
                message: "User authentication failed",
                data: []
            });
        }
        // Fetch user details from database to get branch_id
        const userDetails = await Conveyance.getUserById(userId);
        if (!userDetails || !userDetails.branch_id) {
            return res.status(200).json({
                status: 400,
                success: false,
                message: "User branch information not found",
                data: []
            });
        }
        // Convert comma-separated branch_id to array
        const userBranchIds = userDetails.branch_id.split(',').map(id => parseInt(id.trim())).filter(id => !isNaN(id));

        const { claims, total } = await Conveyance.datatableFinanceClaims({ page, limit, sortBy, sortOrder, search, userBranchIds });
        const totalPages = Math.ceil(total / limit);
        const paginationMeta = {
            total,
            page,
            limit,
            totalPages
        };
        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Finance approved claims fetched successfully',
            data: claims,
            total,
            pagination: paginationMeta
        });
    } catch (error) {
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Error fetching datatable finance claims',
            error: error.message
        });
    }
};

// Helper function to validate date format
function isValidDate(dateString) {
    const regex = /^\d{4}-\d{2}-\d{2}$/;
    if (!regex.test(dateString)) return false;
    
    const date = new Date(dateString);
    return date instanceof Date && !isNaN(date) && date.toISOString().slice(0, 10) === dateString;
}


