import { findAllConveyance, findConveyanceByUserId } from '../models/conveyanceModel.js';

async function getConveyance(req, res) {
    try {
        const { start_date, end_date } = req.query;
        
        // Validate date format if provided
        if (start_date && !isValidDate(start_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid start_date format. Use YYYY-MM-DD"
            });
        }
        
        if (end_date && !isValidDate(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid end_date format. Use YYYY-MM-DD"
            });
        }

        const conveyance = await findAllConveyance(start_date, end_date);
        res.status(200).json(
            {
                status: 200,
                success: true,
                message: "Conveyance list fetched successfully",
                data: conveyance
            }
        );
    } catch (error) {
        res.status(500).json(
            {
                status: 500,
                success: false,
                message: "Error fetching conveyance",
                error: error.message
            }
        );
    }
}

async function getConveyanceByUserId(req, res) {
    try {
        const { id } = req.params;
        const { start_date, end_date } = req.query;
        
        // Validate user ID
        if (!id || isNaN(id)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid user ID"
            });
        }
        
        // Validate date format if provided
        if (start_date && !isValidDate(start_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid start_date format. Use YYYY-MM-DD"
            });
        }
        
        if (end_date && !isValidDate(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid end_date format. Use YYYY-MM-DD"
            });
        }

        const conveyance = await findConveyanceByUserId(id, start_date, end_date);
        res.status(200).json(
            {
                status: 200,
                success: true,
                message: "Conveyance details fetched successfully",
                data: conveyance
            }
        );
    } catch (error) {
        res.status(500).json(
            {
                status: 500,
                success: false,
                message: "Error fetching conveyance by user ID",
                error: error.message
            }
        );
    }
}

// Helper function to validate date format
function isValidDate(dateString) {
    const regex = /^\d{4}-\d{2}-\d{2}$/;
    if (!regex.test(dateString)) return false;
    
    const date = new Date(dateString);
    return date instanceof Date && !isNaN(date) && date.toISOString().slice(0, 10) === dateString;
}

export { 
    getConveyance,
    getConveyanceByUserId
};
