import { db } from '../utils/database.js';

function haversineDistance(lat1, lon1, lat2, lon2) {
    // Haversine formula to calculate distance in km
    function toRad(x) { return x * Math.PI / 180; }
    const R = 6371; // km
    const dLat = toRad(lat2 - lat1);
    const dLon = toRad(lon2 - lon1);
    const a = Math.sin(dLat / 2) * Math.sin(dLat / 2) +
        Math.cos(toRad(lat1)) * Math.cos(toRad(lat2)) *
        Math.sin(dLon / 2) * Math.sin(dLon / 2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    return R * c;
}

export const locationTracking = async (req, res) => {
    try {
        const { user_id, latitude, longitude, location } = req.body;
        if (!user_id || !latitude || !longitude || !location) {
            return res.status(400).json({
                status: false,
                message: 'Validation failed',
                errors: {
                    user_id: !user_id ? 'user_id is required' : undefined,
                    latitude: !latitude ? 'latitude is required' : undefined,
                    longitude: !longitude ? 'longitude is required' : undefined,
                    location: !location ? 'location is required' : undefined
                }
            });
        }

        const date = new Date().toISOString().slice(0, 10);

        // 1. Get attendance config
        let attendanceConfigRows = await db.query('SELECT * FROM attendanceconfigs LIMIT 1');
        const attendanceconfig = attendanceConfigRows[0];

        // 2. Get open attendance task
        let openattendanceRows = await db.query(
            `SELECT * FROM tasks WHERE tasktype_id = 1 AND DATE(checkin_at) = ? AND user_id = ? LIMIT 1`,
            [date, user_id]
        );
        const openattendance = openattendanceRows[0] || null;

        // 3. Get open site visit task (tasktype_id = 3, status Inprogress, soft_checkout = 0)
        let opentaskRows = await db.query(
            `SELECT * FROM tasks WHERE tasktype_id = 3 AND DATE(checkin_at) = ? AND soft_checkout = 0 AND user_id = ? AND status = 'Inprogress' LIMIT 1`,
            [date, user_id]
        );
        const opentask = opentaskRows[0] || null;

        // 4. Get last tracking location
        let lastTrackRows = await db.query(
            `SELECT * FROM locationtrackings WHERE user_id = ? ORDER BY id DESC LIMIT 1`,
            [user_id]
        );
        const lasttracklocation = lastTrackRows[0] || null;

        let deviation = 0;
        let timeDiffInMinutes = 0;
        if (lasttracklocation) {
            const trackingTime = new Date(lasttracklocation.tracking_time);
            const now = new Date();
            timeDiffInMinutes = Math.floor((now - trackingTime) / 60000);
        }

        let response = {};
        if (timeDiffInMinutes >= (attendanceconfig?.capture_time || 0)) {
            if (opentask) {
                const taskcheckinLat = opentask.checkin_lat;
                const taskcheckinLong = opentask.checkin_long;
                const trackcheckinLat = latitude;
                const trackcheckinLong = longitude;

                deviation = haversineDistance(
                    Number(taskcheckinLat),
                    Number(taskcheckinLong),
                    Number(trackcheckinLat),
                    Number(trackcheckinLong)
                );

                if (deviation > (attendanceconfig?.capture_km || 0)) {
                    // Soft checkout
                    await db.query(
                        `UPDATE tasks SET soft_checkout = 1, soft = 1, soft_datetime = NOW(), soft_location = ?, softcheckout_lat = ?, softcheckout_long = ? WHERE id = ?`,
                        [location, latitude, longitude, opentask.id]
                    );

                    // Fetch user device token
                    let userRows = await db.query(`SELECT device_token FROM users WHERE id = ? LIMIT 1`, [opentask.user_id]);
                    const deviceToken = userRows[0]?.device_token || null;

                    // Notification logic (implement your push notification service here)
                    // Notification logic (implement your push notification service here)
                    if (deviceToken) {
                        const message = {
                            notification: {
                                title: 'Soft Checkout Alert',
                                body: `Your current Site Visit was soft Checked out because you had moved ${attendanceconfig.capture_km} km away from the site.`
                            },
                            token: deviceToken
                        };
                        try {
                            await admin.messaging().send(message);
                        } catch (err) {
                            console.error('Firebase push notification error:', err);
                        }
                    }
                    // Example: await pushNotificationService.sendNotification({ ... });

                    // Save to firebaseinbox (if you have such a table)
                    await db.query(
                        `INSERT INTO firebaseinboxs (user_id, category, message, action, task_id, trigger_date, trigger_time, read_status)
                         VALUES (?, ?, ?, ?, ?, CURDATE(), NOW(), 1)`,
                        [
                            opentask.user_id,
                            `Soft Checkout reminder-${opentask.task}`,
                            `Your current Site Vist was soft Checked out because you had moved ${attendanceconfig.capture_km} km away from the site.`,
                            2,
                            opentask.id
                        ]
                    );
                }
            }

            // Save new tracking
            await db.query(
                `INSERT INTO locationtrackings (user_id, latitude, longitude, location, tracking_date, tracking_time, distance)
                 VALUES (?, ?, ?, ?, CURDATE(), NOW(), ?)`,
                [user_id, latitude, longitude, location, deviation]
            );

            response = {
                user_id,
                status: true,
                task: opentask,
                attendance: openattendance
            };
        } else if (timeDiffInMinutes === 0) {
            // First tracking of the day
            await db.query(
                `INSERT INTO locationtrackings (user_id, latitude, longitude, location, tracking_date, tracking_time, distance)
                 VALUES (?, ?, ?, ?, CURDATE(), NOW(), ?)`,
                [user_id, latitude, longitude, location, deviation]
            );
        } else {
            if (timeDiffInMinutes >= (attendanceconfig?.capture_time || 0)) {
                await db.query(
                    `INSERT INTO locationtrackings (user_id, latitude, longitude, location, tracking_date, tracking_time, distance)
                     VALUES (?, ?, ?, ?, CURDATE(), NOW(), ?)`,
                    [user_id, latitude, longitude, location, deviation]
                );
                response = {
                    user_id,
                    status: true,
                    task: opentask,
                    attendance: openattendance
                };
            }
        }

        // Always return the latest status
        response = {
            user_id,
            task: opentask,
            attendance: openattendance
        };

        return res.status(200).json({
            status: true,
            message: 'Save location tracking success',
            response
        });
    } catch (error) {
        console.error('Location tracking error:', error);
        return res.status(500).json({
            status: false,
            message: 'Error saving location tracking',
            response: {}
        });
    }
};