import { Project } from '../models/project.js';
import { db } from '../utils/database.js';

export const createProject = async (req, res) => {
    try {
        const projectId = await Project.create(req.body, req.user.id);
        const project = await Project.findById(projectId);
        
        res.status(201).json({
            status: 201,
            message: 'Project created successfully',
            project
        });
    } catch (error) {
        console.error('Create project error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to create project'
        });
    }
};

export const updateProject = async (req, res) => {
    try {
        const { id } = req.params;
        await Project.update(id, req.body);
        const project = await Project.findById(id);

        res.json({
            status: 200,
            message: 'Project updated successfully',
            project
        });
    } catch (error) {
        console.error('Update project error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to update project'
        });
    }
};

export const deleteProject = async (req, res) => {
    try {
        const { id } = req.params;
        await Project.softDelete(id);

        res.json({
            status: 200,
            message: 'Project deleted successfully'
        });
    } catch (error) {
        console.error('Delete project error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to delete project'
        });
    }
};

export const listProjects = async (req, res) => {
    try {
        const userId = req.user.id;
        const { start_date, end_date } = req.query;
        const projects = await Project.list(userId, start_date, end_date);
        res.json({
            status: 200,
            projects
        });
    } catch (error) {
        console.error('List projects error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to list projects'
        });
    }
};

export const getAssignedProjects = async (req, res) => {
    try {
        const userId = req.user.id;
        console.log(req.query);
        const result = await Project.getAssignedProjects(userId, req.query);
        console.log();
        res.json({
            code: 200,
            status: true,
            message: "Assigned projects fetched successfully",
            response: result
        });
    } catch (error) {
        res.json({
            code: 500,
            status: false,
            message: error.message || "Failed to fetch assigned projects"
        });
    }
};

export const getProjectDetails = async (req, res) => {
    try {
        const { id } = req.params;
        const userId = req.user.id;

        const result = await Project.getProjectDetailsWithTasks(id, userId);
        console.log(result);
        res.json({
            code: 200,
            status: true,
            message: "Project details retrieved successfully",
            response: result
        });
    } catch (error) {
        console.error('Get project details error:', error);
        res.json({
            code: error.message === 'Project not found' ? 404 : 500,
            status: false,
            message: error.message || 'Failed to retrieve project details'
        });
    }
};
export const getProjectAdminDetails = async (req, res) => {
    try {
        const { id } = req.params;
        const userId = req.user.id;

        const result = await Project.getProjectDetailsWithAdminTasks(id, userId);
        console.log(result);
        res.json({
            code: 200,
            status: true,
            message: "Project details retrieved successfully",
            response: result
        });
    } catch (error) {
        console.error('Get project details error:', error);
        res.json({
            code: error.message === 'Project not found' ? 404 : 500,
            status: false,
            message: error.message || 'Failed to retrieve project details'
        });
    }
};

export const projectFilterList = async (req, res) => {
    const response = {};
    try {
        const userId = req.user.id;

        const result = await Project.projectFilterList(userId);

        if (result.notExists) {
            return res.status(200).json({
                code: 204,
                status: false,
                message: 'No projects available',
                response
            });
        }

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Projects filter details',
            response: result
        });
    } catch (error) {
        console.error('Project filter list error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'Error - ' + error.message,
            response
        });
    }
};

// Helper function to validate date format
const isValidDate = (dateString) => {
    const regex = /^\d{4}-\d{2}-\d{2}$/;
    if (!regex.test(dateString)) return false;
    
    const date = new Date(dateString);
    return date instanceof Date && !isNaN(date);
};

export const updateProjectStatus = async (req, res) => {
    try {
        const { id } = req.params;
        const { project_status } = req.body;

        if (!id || !project_status) {
            return res.status(400).json({
                code: 400,
                status: false,
                message: 'Project ID and project_status are required',
                response: {}
            });
        }

        let query, params;
        if (project_status.toLowerCase() === 'closed') {
            query = `
                UPDATE projects
                SET project_status = ?, closure_date = CURDATE(), updated_at = NOW()
                WHERE id = ? AND status = 1
            `;
            params = [project_status, id];
        } else {
            query = `
                UPDATE projects
                SET project_status = ?, updated_at = NOW()
                WHERE id = ? AND status = 1
            `;
            params = [project_status, id];
        }

        await db.query(query, params);

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Project status updated successfully',
            response: {}
        });
    } catch (error) {
        console.error('Update project status error:', error);
        return res.status(500).json({
            code: 500,
            status: false,
            message: 'Failed to update project status',
            response: {}
        });
    }
};

// Add Warranty API
export const addProjectWarranty = async (req, res) => {
    try {
        const { project_id, product_id, warranty_start_date, warranty_end_date, notes } = req.body;
        const created_by = req.user.id;
        const create_date = new Date().toISOString().slice(0, 10);

        if (!project_id || !product_id || !warranty_start_date || !warranty_end_date) {
            return res.status(400).json({
                code: 400,
                status: false,
                message: 'project_id, product_id, warranty_start_date, and warranty_end_date are required',
                response: {}
            });
        }

        await db.query(
            `INSERT INTO projectwarranties 
                (project_id, product_id, warranty_start_date, warranty_end_date, notes, create_date, created_by, created_at, updated_at)
             VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), NOW())`,
            [
                project_id,
                product_id,
                warranty_start_date,
                warranty_end_date,
                notes || '',
                create_date,
                created_by
            ]
        );

        return res.status(201).json({
            code: 201,
            status: true,
            message: 'Warranty added successfully',
            response: {}
        });
    } catch (error) {
        console.error('Add warranty error:', error);
        return res.status(500).json({
            code: 500,
            status: false,
            message: 'Failed to add warranty',
            response: {}
        });
    }
};

// Edit Warranty API
export const editProjectWarranty = async (req, res) => {
    try {
        const { id } = req.params;
        const { project_id, product_id, warranty_start_date, warranty_end_date, notes } = req.body;
        const updated_by = req.user.id;

        if (!id || !project_id || !product_id || !warranty_start_date || !warranty_end_date) {
            return res.status(400).json({
                code: 400,
                status: false,
                message: 'id, project_id, product_id, warranty_start_date, and warranty_end_date are required',
                response: {}
            });
        }

        await db.query(
            `UPDATE projectwarranties 
             SET project_id = ?, product_id = ?, warranty_start_date = ?, warranty_end_date = ?, notes = ?, updated_at = NOW()
             WHERE id = ?`,
            [
                project_id,
                product_id,
                warranty_start_date,
                warranty_end_date,
                notes || '',
                id
            ]
        );

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Warranty updated successfully',
            response: {}
        });
    } catch (error) {
        console.error('Edit warranty error:', error);
        return res.status(500).json({
            code: 500,
            status: false,
            message: 'Failed to update warranty',
            response: {}
        });
    }
};

export const deleteProjectWarranty = async (req, res) => {
    try {
        const { id } = req.params;
        if (!id) {
            return res.status(400).json({
                status: 400,
                message: 'Warranty ID is required'
            });
        }

        // Assuming you have a Project model method to delete warranty
        const result = await Project.deleteWarrantyById(id);

        if (result.affectedRows === 0) {
            return res.status(404).json({
                status: 404,
                message: 'Warranty not found'
            });
        }

        res.status(200).json({
            status: 200,
            message: 'Warranty deleted successfully'
        });
    } catch (error) {
        console.error('Delete warranty error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to delete warranty'
        });
    }
};

export const getProjectProducts = async (req, res) => {
    try {
        const { id } = req.params;
        if (!id) {
            return res.status(400).json({
                status: 400,
                message: 'Project ID is required'
            });
        }

        const query = `
            SELECT 
                pp.id as projectpackage_id,
                pk.id as package_id,
                pk.package_name,
                pr.id as product_id,
                pr.name,
                pr.sku_no
            FROM projectpackages pp
            LEFT JOIN packages pk ON pp.package_id = pk.id
            LEFT JOIN packageproducts ppr ON pk.id = ppr.package_id
            LEFT JOIN products pr ON ppr.product_id = pr.id
            WHERE pp.project_id = ?
        `;
        let products = await db.query(query, [id]);

        res.status(200).json({
            status: 200,
            message: 'Project products fetched successfully',
            data: products
        });
    } catch (error) {
        console.error('Fetch project products error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to fetch project products'
        });
    }
};

export const viewProjectWithPackages = async (req, res) => {
    try {
        const { projectId } = req.params;
        const project = await Project.getProjectWithPackages(projectId);
        res.json({ status: true, project });
    } catch (error) {
        res.status(500).json({ status: false, message: error.message });
    }
};