import { ProjectModule } from '../models/projectModule.js';
import { db } from '../utils/database.js';

export class ProjectModuleController {
    static async create(req, res) {
        try {
            const { name, description } = req.body;
            
            if (!name) {
                return res.status(400).json({ 
                    success: false, 
                    message: 'Name is required' 
                });
            }
            
            const [result] = await db.query(
                'INSERT INTO project_modules (name, description) VALUES (?, ?)',
                [name, description]
            );
            
            res.status(201).json({
                success: true,
                data: { id: result.insertId, name, description }
            });
            
        } catch (error) {
            console.error('Error creating project module:', error);
            res.status(500).json({ 
                success: false, 
                message: 'Failed to create project module',
                error: error.message 
            });
        }
    }

    static async getAll(req, res) {
        try {
            const [results] = await db.query('SELECT * FROM project_modules');
            
            res.status(200).json({
                success: true,
                data: results
            });
            
        } catch (error) {
            console.error('Error getting project modules:', error);
            res.status(500).json({ 
                success: false, 
                message: 'Failed to get project modules',
                error: error.message 
            });
        }
    }

    static async getById(req, res) {
        try {
            const { id } = req.params;
            
            const [results] = await db.query(
                'SELECT * FROM project_modules WHERE id = ?',
                [id]
            );
            
            if (!results || results.length === 0) {
                return res.status(404).json({ 
                    success: false, 
                    message: 'Project module not found' 
                });
            }
            
            res.status(200).json({
                success: true,
                data: results[0]
            });
            
        } catch (error) {
            console.error('Error getting project module:', error);
            res.status(500).json({ 
                success: false, 
                message: 'Failed to get project module',
                error: error.message 
            });
        }
    }

    static async update(req, res) {
        try {
            const { id } = req.params;
            const { name, description } = req.body;
            
            if (!name) {
                return res.status(400).json({ 
                    success: false, 
                    message: 'Name is required' 
                });
            }
            
            const [result] = await db.query(
                'UPDATE project_modules SET name = ?, description = ? WHERE id = ?',
                [name, description, id]
            );
            
            if (result.affectedRows === 0) {
                return res.status(404).json({ 
                    success: false, 
                    message: 'Project module not found' 
                });
            }
            
            res.status(200).json({
                success: true,
                message: 'Project module updated successfully'
            });
            
        } catch (error) {
            console.error('Error updating project module:', error);
            res.status(500).json({ 
                success: false, 
                message: 'Failed to update project module',
                error: error.message 
            });
        }
    }

    static async delete(req, res) {
        try {
            const { id } = req.params;
            
            const [result] = await db.query(
                'DELETE FROM project_modules WHERE id = ?',
                [id]
            );
            
            if (result.affectedRows === 0) {
                return res.status(404).json({ 
                    success: false, 
                    message: 'Project module not found' 
                });
            }
            
            res.status(200).json({
                success: true,
                message: 'Project module deleted successfully'
            });
            
        } catch (error) {
            console.error('Error deleting project module:', error);
            res.status(500).json({ 
                success: false, 
                message: 'Failed to delete project module',
                error: error.message 
            });
        }
    }
}