import { Task } from '../models/task.js';
import { db } from '../utils/database.js';

export class ReportController {
    static async getAllActionReport(req, res) {
        try {
            const { userId, startDate, endDate } = req.query;
            
            // Validate required parameters
            if (!userId || !startDate || !endDate) {
                return res.status(400).json({ 
                    success: false, 
                    message: 'Missing required parameters: userId, startDate, or endDate' 
                });
            }
            
            // Query database for all actions within date range
            const query = `
                SELECT 
                    t.id as task_id,
                    t.task,
                    t.description,
                    t.start_date,
                    t.start_time,
                    t.status,
                    tt.activity_name as action_type,
                    p.project_id,
                    p.project_name,
                    c.company_name,
                    ct.contact_name
                FROM tasks t
                LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
                LEFT JOIN projects p ON t.project_id = p.id
                LEFT JOIN companies c ON p.customer_id = c.id
                LEFT JOIN contacts ct ON p.contact_id = ct.id
                WHERE t.user_id = ? 
                AND DATE(t.start_date) BETWEEN ? AND ?
                ORDER BY t.start_date DESC, t.start_time DESC
            `;
            
            const [results] = await db.query(query, [userId, startDate, endDate]);
            
            if (!results || results.length === 0) {
                return res.status(404).json({ 
                    success: false, 
                    message: 'No actions found for the specified criteria' 
                });
            }
            
            res.status(200).json({
                success: true,
                data: results
            });
            
        } catch (error) {
            console.error('Error in getAllActionReport:', error);
            res.status(500).json({ 
                success: false, 
                message: 'Internal server error',
                error: error.message 
            });
        }
    }
}