import { TaskType } from '../models/tasktype.js';
import { Task } from '../models/task.js';
import { db } from '../utils/database.js';
// Use multer for file uploads
import multer from 'multer';
const upload = multer({ dest: 'public/claim/' }); // configure as needed
import axios from 'axios';
import { json } from 'express';
import { transcribeAudio } from '../utils/speechToText.js';
import { extractTaskDetails } from '../utils/aiExtractor.js'; // You must implement this
import { parse, format } from 'date-fns';
import { log } from 'console';

export const createTaskType = async (req, res) => {
    try {
        // Get the uploaded file path
        const iconPath = req.file ? req.file.path.replace(/\\/g, '/') : null;
        
        if (!iconPath) {
            return res.status(400).json({
                status: 400,
                message: 'Activity icon is required'
            });
        }

        const taskTypeData = {
            ...req.body,
            activity_icon: iconPath
        };

        const taskTypeId = await TaskType.create(taskTypeData);
        const taskType = await TaskType.findById(taskTypeId);
        
        res.status(201).json({
            status: 201,
            message: 'Task type created successfully',
            taskType
        });
    } catch (error) {
        console.error('Create task type error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to create task type'
        });
    }
};

export const updateTaskType = async (req, res) => {
    try {
        const { id } = req.params;
        
        // Get the current task type to check existing icon
        const existingTaskType = await TaskType.findById(id);
        if (!existingTaskType) {
            return res.status(404).json({
                status: 404,
                message: 'Task type not found'
            });
        }

        // Update icon path if new file is uploaded
        const iconPath = req.file ? req.file.path.replace(/\\/g, '/') : existingTaskType.activity_icon;

        const taskTypeData = {
            ...req.body,
            activity_icon: iconPath
        };

        await TaskType.update(id, taskTypeData);
        const taskType = await TaskType.findById(id);

        res.json({
            status: 200,
            message: 'Task type updated successfully',
            taskType
        });
    } catch (error) {
        console.error('Update task type error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to update task type'
        });
    }
};

export const deleteTaskType = async (req, res) => {
    try {
        const { id } = req.params;
        await TaskType.softDelete(id);

        res.json({
            status: 200,
            message: 'Task type deleted successfully'
        });
    } catch (error) {
        console.error('Delete task type error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to delete task type'
        });
    }
};

export const listTaskTypes = async (req, res) => {
    try {
        const taskTypes = await TaskType.list();
        
        res.json({
            status: 200,
            message: 'Task types retrieved successfully',
            taskTypes
        });
    } catch (error) {
        console.error('List task types error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to retrieve task types'
        });
    }
};

export const createAttendanceTasks = async (req, res) => {
    try {
        const tasks = await Task.createAttendanceTasks();
        // console.log("12345678------",tasks);
        
        res.status(200).json({
            // code: 200,
            // status: true,
            // message: "Attendance tasks created successfully",
            // response: {
            //     tasks: tasks
            // }
            tasks
        });
    } catch (error) {
        console.error('Create attendance tasks error:', error);
        res.json({
            code: 500,
            status: false,
            message: 'Failed to create attendance tasks'
        });
    }
};

export const createTask = async (req, res) => {
    try {
        const userId = req.user.id;
        const taskData = req.body;

        // List all possible fields for a task
        const fields = [
            'project_id',
            'task',
            'description',
            'start_date',
            'start_time',
            'end_date',
            'end_time',
            'tasktype_id',
            'tasksubcategory_id',
            'product_id',
            'taskcategoryoption_id',
            // add any other fields your Task.create expects
        ];

        // Ensure all fields are present and set to null if missing or empty
        const cleanTaskData = {};
        for (const field of fields) {
            cleanTaskData[field] =
                (taskData[field] !== undefined && taskData[field] !== '') ? taskData[field] : null;
        }

        const taskId = await Task.create(cleanTaskData, userId);
        const task = await Task.findById(taskId);

        res.json({
            code: 200,
            status: true,
            message: 'Task created successfully',
            response: task
        });
    } catch (error) {
        console.error('Create task error:', error);
        res.json({
            code: 500,
            status: false,
            message: error.message || 'Failed to create task'
        });
    }
};

export const updateTask = async (req, res) => {
    try {
        const { id } = req.params;
        const taskData = req.body;

        // List all possible fields for a task (same as in createTask)
        const fields = [
            'project_id',
            'task',
            'description',
            'start_date',
            'start_time',
            'end_date',
            'end_time',
            'tasktype_id',
            'tasksubcategory_id',
            'product_id',
            'taskcategoryoption_id',
            // add any other fields your Task.update expects
        ];

        // Ensure all fields are present and set to null if missing or empty
        const cleanTaskData = {};
        for (const field of fields) {
            cleanTaskData[field] =
                (taskData[field] !== undefined && taskData[field] !== '') ? taskData[field] : null;
        }

        await Task.update(id, cleanTaskData);
        const task = await Task.findById(id);

        res.json({
            code: 200,
            status: true,
            message: 'Task updated successfully',
            response: task
        });
    } catch (error) {
        console.error('Update task error:', error);
        res.json({
            code: 500,
            status: false,
            message: 'Failed to update task'
        });
    }
};

export const getUserTaskDashboard = async (req, res) => {
    try {
        const userId = req.user.id;
        const { date, device_id } = req.query;

        if (!date || !device_id) {
            return res.json({
                code: 400,
                status: false,
                message: 'date and device_id are required'
            });
        }

        const result = await Task.getUserDashboard(userId, date, device_id);

        if (!result.isDeviceMatch) {
            return res.status(401).json({
                code: 401,
                status: false,
                message: 'Device ID does not match'
            });
        }

        // Ensure response structure as per prompt
        res.json({
            code: 200,
            status: true,
            message: 'User task dashboard fetched successfully',
            response: {
                tasks: Array.isArray(result.tasks) ? result.tasks : [],
                opentasklist: Array.isArray(result.inProgressTasks) ? result.inProgressTasks : [],
                attendanceconfig: result.attendanceConfig || null,
                attendance: result.attendance || null,
                branch: result.branch || null,
                isCheckinCheckout: !!result.isCheckinCheckout,
                change_password: result.change_password || null
            }
        });
    } catch (error) {
        console.error('Get user task dashboard error:', error);
        res.json({
            code: 500,
            status: false,
            message: 'Failed to fetch user task dashboard'
        });
    }
};

export const changeTaskStatus = async (req, res) => {
    try {
        const { id } = req.params;

        // Fetch current task
        const task = await Task.findById(id);
        if (!task) {
            return res.status(404).json({
                code: 404,
                status: false,
                message: 'Task not found'
            });
        }

        let newStatus = null;
        if (task.status === 'Open') {
            newStatus = 'Inprogress';
        } else if (task.status === 'Inprogress') {
            newStatus = 'Closed';
        } else {
            return res.json({
                code: 200,
                status: true,
                message: `Task status is already ${task.status}`,
                response: { status: task.status }
            });
        }

        await Task.updateStatus(id, { status: newStatus });

        res.json({
            code: 200,
            status: true,
            message: `Task status updated to ${newStatus}`,
            response: { status: newStatus }
        });
    } catch (error) {
        console.error('Change task status error:', error);
        res.status(500).json({
            code: 500,
            status: false,
            message: 'Failed to change task status'
        });
    }
};

export const updateTaskStatus = async (req, res) => {
    try {
        const { id } = req.params;
        const { latitude, longitude, location, remarks, transport_mode, claim_amount, claim_remarks, ack_contact_id, actual_checkout_date } = req.body;
        const file = req.file;
        const claim_image = req.files && req.files.claim_image ? req.files.claim_image[0] : null; // If using multer.fields
        const uploadedFiles = req.files && req.files.task_files ? req.files.task_files : []; // Expecting 'task_files' as field name for multiple files
        const userId = req.user.id;

        // Fetch current task
        const task = await Task.findById(id);
        if (!task) {
            return res.status(404).json({
                code: 404,
                status: false,
                message: 'Task not found'
            });
        }
        console.log("task id"+id);

        console.log("task details"+task);
        // --- Add this block to update user's current location ---
        if (latitude && longitude) {
            await db.query(
                `UPDATE users SET current_latitude = ?, current_longitude = ? WHERE id = ?`,
                [latitude, longitude, userId]
            );
        }
        // --- End block ---

        let updateData = {};
        const now = new Date();
        console.log("task status"+task.status);
        if (task.status === 'Open') {
            // Get previous task for this user (by checkin_at desc)
            const prevTask = await Task.getPreviousTask(userId, id);

            let distanceTravelled = 0;
            console.log("previous task"+JSON.stringify(prevTask));
            if (prevTask) {
                if (prevTask.tasktype_id === 1) {
                    // Fetch user's branch latitude/longitude
                    let branchRows = await db.query(
                        `SELECT b.branch_lat, b.branch_long 
                         FROM users u 
                         LEFT JOIN branches b ON u.branch_id = b.id 
                         WHERE u.id = ? LIMIT 1`, [userId]
                    );
                    console.log('userId:', userId);
                    let branch = branchRows && branchRows[0] ? branchRows[0] : null;
                    console.log('branchRows:', branchRows);
                    if (branchRows && branchRows[0]) {
                        branch = branchRows[0];
                    }

                    // Fetch user's home latitude/longitude
                    let userRows = await db.query(
                        `SELECT home_latitude, home_longitude, home_approval FROM users WHERE id = ? LIMIT 1`, [userId]
                    );
                    const userHome = userRows && userRows[0] ? userRows[0] : null;

                    // Calculate distance from branch to current, and home to current
                    let branchDistance = null, homeDistance = null, prevTaskDistance = null, prevTaskDistance1 = null;
                    if (branch && branch.branch_lat && branch.branch_long) {
                        branchDistance = await Task.calculateDistance(
                            branch.branch_lat,
                            branch.branch_long,
                            latitude,
                            longitude
                        );
                    }
                    if (prevTask && prevTask.checkin_lat && prevTask.checkin_long) {
                        prevTaskDistance = await Task.calculateDistance(
                            prevTask.checkin_lat,
                            prevTask.checkin_long,
                            branch.branch_lat,
                            branch.branch_long
                        );
                        prevTaskDistance1 = await Task.calculateDistance(
                            prevTask.checkin_lat,
                            prevTask.checkin_long,
                            latitude,
                            longitude
                        );
                    }
                    if ((userHome && userHome.home_latitude && userHome.home_longitude)||(userHome.home_approval == "Approved")) {
                        homeDistance = await Task.calculateDistance(
                            userHome.home_latitude,
                            userHome.home_longitude,
                            latitude,
                            longitude
                        );
                    }
                    console.log("branchDistance: " + branchDistance);
                    console.log("homeDistance: " + homeDistance);
                    // Save the shortest non-null distance
                    if(prevTaskDistance <= 0.5) {
                        prevTaskDistance1 = branchDistance;
                    }
                    else if (branchDistance !== null && homeDistance !== null) {
                        distanceTravelled = Math.min(branchDistance, homeDistance);
                    } else if (branchDistance !== null) {
                        distanceTravelled = branchDistance;
                    } else if (homeDistance !== null) {
                        distanceTravelled = homeDistance;
                    } else {
                        distanceTravelled = 0;
                    }
                } else {
                    // For other tasktype_id, use last task's checkin_lat/long
                    if (prevTask.checkin_lat && prevTask.checkin_long) {
                        distanceTravelled = await Task.calculateDistance(
                            prevTask.checkin_lat,
                            prevTask.checkin_long,
                            latitude,
                            longitude
                        );
                    }
                }
            }

            updateData = {
                status: 'Inprogress',
                checkin_lat: latitude,
                checkin_long: longitude,
                checkin_location: location,
                checkin_at: now,
                distance_travelled: distanceTravelled,
                ...(remarks && { remarks })
            };
        } 
        else if (task.status === 'Inprogress') {
            let timespent = null;
            if (task.checkin_at) {
                const checkinAt = new Date(task.checkin_at);
                timespent = Math.floor((now - checkinAt) / 1000);
            }
            
            let distanceTravelled=0;
            //let distanceTravelled = task.distance_travelled || 0;
            console.log(task.tasktype_id);
            
            // Fetch user's home lat/long and role info (moved outside conditional for all task types)
            console.log('userId:', userId);
            let userRows = await db.query(
                `SELECT u.home_latitude, u.home_longitude, u.roles_id, r.default_transport 
                 FROM users u 
                 LEFT JOIN roles r ON u.roles_id = r.id 
                 WHERE u.id = ? LIMIT 1`, [userId]
            );
            
            const userHome = userRows && userRows[0] ? userRows[0] : null;
            console.log('User data with role:', userHome);
            
            // Calculate distance for attendance task as before (Google API logic)
            if (task.tasktype_id === 1 && task.checkin_lat && task.checkin_long) {
                console.log("attendance");
                console.log(task.tasktype_id);
                const prevTask = await Task.getNextTask(userId, id);
                console.log("prevtask"+prevTask);
                // Fetch user's branch lat/long
                let branchRows = await db.query(
                    `SELECT b.branch_lat, b.branch_long 
                     FROM users u 
                     LEFT JOIN branches b ON u.branch_id = b.id 
                     WHERE u.id = ? LIMIT 1`, [userId]
                );
                console.log(branchRows);
                const branch = branchRows && branchRows[0] ? branchRows[0] : null;
                // Prepare Google Maps Distance Matrix API call
                /*const origins = `${task.checkin_lat},${task.checkin_long}`;
                let destinations = [];
                if (branch && branch.branch_lat && branch.branch_long) {
                    destinations.push(`${branch.branch_lat},${branch.branch_long}`);
                }
                if (userHome && userHome.home_latitude && userHome.home_longitude) {
                    destinations.push(`${userHome.home_latitude},${userHome.home_longitude}`);
                }*/
                
                let branchDistance = null, homeDistance = null, prevTaskDistance = null, prevTaskDistance1 = null;
                if (branch && branch.branch_lat && branch.branch_long) {

                console.log(branch.branch_long);
                console.log("task checkin");
                console.log("task checkin"+prevTask.checkin_lat);
                console.log("task checkin"+prevTask.checkin_long);
                     branchDistance = await Task.calculateDistance(
                        prevTask.checkin_lat,
                        prevTask.checkin_long,
                        branch.branch_lat,
                        branch.branch_long
                    );
                }
                if (prevTask && prevTask.checkin_lat && prevTask.checkin_long) {
                   /*prevTaskDistance = await Task.calculateDistance(
                        prevTask.checkin_lat,
                        prevTask.checkin_long,
                        branch.branch_lat,
                        branch.branch_long
                    );*/
                    prevTaskDistance1 = await Task.calculateDistance(
                        prevTask.checkin_lat,
                        prevTask.checkin_long,
                        latitude,
                        longitude
                    );
                }
                if (userHome && userHome.home_latitude && userHome.home_longitude) {
                    console.log(userHome.home_latitude);
                 homeDistance = await Task.calculateDistance(
                        prevTask.checkin_lat,
                        prevTask.checkin_long,
                        userHome.home_latitude,
                        userHome.home_longitude,
                    );
                }
                console.log("branchDistance: " + branchDistance);
                console.log("homeDistance: " + homeDistance);
                if(branchDistance <= 0.5) {
                    console.log("firstif: " + branchDistance);
                    distanceTravelled = branchDistance;
                }
                else if (branchDistance !== null && homeDistance !== null) {
                    
                    console.log("secondif: " + branchDistance);
                    distanceTravelled = Math.min(branchDistance, homeDistance);
                } else if (branchDistance !== null) {
                    console.log("thirdif: " + branchDistance);
                    distanceTravelled = branchDistance;
                } else if (homeDistance !== null) {
                    console.log("forthif: " + homeDistance);
                    distanceTravelled = homeDistance;
                } else {
                    console.log("fifthif: 0");
                    distanceTravelled = 0;
                }
                /*if (destinations.length > 0) {
                    const apiKey = process.env.GOOGLE_MAPS_API_KEY;
                    const url = `https://maps.googleapis.com/maps/api/distancematrix/json?origins=${origins}&destinations=${destinations.join('|')}&key=${apiKey}`;
                    const response = await axios.get(url);
                    const elements = response.data.rows[0].elements;

                    // Get the shortest distance in meters
                    let minDistance = null;
                    elements.forEach(el => {
                        if (el.status === 'OK') {
                            if (minDistance === null || el.distance.value < minDistance) {
                                minDistance = el.distance.value;
                            }
                        }
                    });
                    console.log(minDistance);
                    // Convert meters to kilometers
                    distanceTravelled = minDistance !== null ? (minDistance / 1000) : 0;
                }*/
                    console.log("tasktypeid distance: " + distanceTravelled);
            } else {
                // For other task types, fallback to haversine or your own method
                distanceTravelled = task.distance_travelled || 0;
            }
            
            console.log("outside distance: " + distanceTravelled);
            
            // Claim calculation logic
            let claimImageFileName = null;
            if (transport_mode === 'Public Transport' && claim_image) {
                claimImageFileName = claim_image.filename;
            }

            let conveyance_charge = 0;
            let finalClaimAmount = claim_amount || 0;
            
            // Determine transport mode - use provided transport_mode or default from role
            let effectiveTransportMode = transport_mode;
            if (!effectiveTransportMode && userHome && userHome.default_transport) {
                effectiveTransportMode = userHome.default_transport;
                console.log('Using default transport from role:', effectiveTransportMode);
            }
            
            // Calculate claim amount based on transport mode and distance
            if (effectiveTransportMode && effectiveTransportMode !== 'Public Transport') {
                // Fetch conveyance charge for the transport_mode
                let configRows = await db.query(
                    'SELECT conveyance_charge FROM conveyanceconfigs WHERE transport_mode = ? LIMIT 1',
                    [effectiveTransportMode]
                );
                conveyance_charge = configRows && configRows[0] ? configRows[0].conveyance_charge : 0;
                finalClaimAmount = conveyance_charge * distanceTravelled;
                console.log(`Claim calculation: ${conveyance_charge} × ${distanceTravelled} = ${finalClaimAmount}`);
            } else if (effectiveTransportMode === 'Public Transport') {
                // For public transport, claim amount is typically fixed or based on receipt
                finalClaimAmount = claim_amount || 0;
                console.log('Public transport claim amount:', finalClaimAmount);
            } else {
                // No transport mode specified, no claim
                finalClaimAmount = 0;
                console.log('No transport mode specified, claim amount: 0');
            }
            
            updateData = {
                status: 'Closed',
                checkout_lat: latitude,
                checkout_long: longitude,
                checkout_location: location,
                checkout_at: now,
                file_name: file ? file.filename : null,
                time_spent: timespent,
                distance_travelled: distanceTravelled,
                transport_mode: effectiveTransportMode  || null,
                claim_amount: finalClaimAmount || null,
                claim_remarks: claim_remarks || null,
                ack_contact_id: ack_contact_id || null,
                manual_checkout_at: actual_checkout_date || null,
                ...(remarks && { remarks })
            };
        } else {
            return res.json({
                code: 200,
                status: true,
                message: `Task status is already ${task.status}`,
                response: { status: task.status }
            });
        }

        // Save claim image in claimuploads table if transport_mode is Public Transport and claim_image is present
        let claimUploadId = null;
        if (transport_mode === 'Public Transport' && claim_image) {
            await db.query(
                `INSERT INTO claimuploads (task_id, uploadfile, created_at, updated_at)
                 VALUES (?, ?, NOW(), NOW())`,
                [id, claim_image.filename] // claim_image.filename includes the extension
            );
        }
        if (req.files && req.files.voicenote && req.files.voicenote[0]) {
            const voicenoteFile = req.files.voicenote[0];
            // Fetch project_id from the task table
            let taskRow = await db.query(
                'SELECT project_id FROM tasks WHERE id = ? LIMIT 1',
                [id]
            );
            const project_id = taskRow && taskRow[0] ? taskRow[0].project_id : null;

            await db.query(
                `INSERT INTO tasknotes (project_id, task_id, user_id, notes, file, file_name, created_at, updated_at, create_date)
                 VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW(), NOW())`,
                [project_id, id, req.user.id, '', voicenoteFile.filename, voicenoteFile.filename]
            );
        }
        // Store multiple uploaded files in taskuploads table and handle taskactions
        if (uploadedFiles && uploadedFiles.length > 0) {
            for (const f of uploadedFiles) {
                // Insert file into taskuploads table
                let result = await db.query(
                    `INSERT INTO taskuploads (task_id, file_name, created_at, updated_at)
                     VALUES (?, ?, NOW(), NOW())`,
                    [id, f.filename]
                );
                const taskupload_id = result.insertId;

                // Extract the base name (without extension) from original filename
                const originalName = f.originalname;
                const baseName = originalName.split('.')[0]; // e.g., 'desk' from 'desk.png'
                let action_ids = req.body[baseName];
                console.log(`Base name: ${baseName}, Action IDs:`, action_ids);
                // Normalize action_ids to always be an array of numbers
                if (typeof action_ids === 'string') {
                    // Handles both single value "50" and comma-separated "50,51"
                    action_ids = action_ids.split(',').map(id => id.trim()).filter(Boolean);
                } else if (Array.isArray(action_ids)) {
                    // Handles cases like ['[50]'] or ['50', '51']
                    action_ids = action_ids.flatMap(id => {
                        if (typeof id === 'string' && id.startsWith('[') && id.endsWith(']')) {
                            // Remove brackets and split by comma
                            return id.slice(1, -1).split(',').map(x => x.trim()).filter(Boolean);
                        }
                        return id;
                    });
                } else if (action_ids !== undefined && action_ids !== null) {
                    action_ids = [action_ids];
                } else {
                    action_ids = [];
                }

                // Convert all to numbers and filter out invalids
                action_ids = action_ids.map(id => Number(id)).filter(id => !isNaN(id) && id > 0);

                console.log(`Processing file: ${originalName}, sanitized action_ids:`, action_ids);

                for (let action_id of action_ids) {
                    console.log(`Processing action_id: ${action_id}`);
                    // Fetch action_name from taskactionlists table
                    let actionRows = await db.query(
                        'SELECT action_item FROM taskactionlists WHERE id = ? LIMIT 1',
                        [action_id]
                    );
                    const action_name = actionRows && actionRows[0] ? actionRows[0].action_item : null;

                    await db.query(
                        `INSERT INTO taskactions 
                            (remarks, project_id, task_id, action_id, action_name, taskupload_id, created_at, updated_at)
                         VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())`,
                        [
                            remarks || null,
                            task.project_id,
                            task.id,
                            action_id,
                            action_name,
                            taskupload_id
                        ]
                    );
                }
            }
        }

        await Task.updateTaskStatus(id, updateData);
        const taskdata = await Task.findById(id);

        res.json({
            code: 200,
            status: true,
            message: 'Task status updated successfully',
            response: {
                tasks: taskdata,
                ...(claimUploadId && { claim_upload_id: claimUploadId })
            }
        });
    } catch (error) {
        console.error('Update task status error:', error);
        res.status(500).json({
            code: 500,
            status: false,
            message: 'Failed to update task status'
        });
    }
};
export const updateOtherTaskStatus = async (req, res) => {
 try{
        const { id } = req.params;
        console.log(req.params);
        console.log(req.body);

        const { latitude, longitude, location, remarks, transport_mode, claim_amount, claim_remarks, ack_contact_id, actual_checkout_date } = req.body;
        const userId = req.user.id;

        // Fetch current task
        const task = await Task.findById(id);
        console.log(task);
        if (!task) {
            return res.status(404).json({
                code: 404,
                status: false,
                message: 'Task not found'
            });
        }

        console.log("latitude"+latitude);
  
        console.log("longitude"+longitude);
        // --- Add this block to update user's current location ---
        if (latitude && longitude) {
            await db.query(
                `UPDATE users SET current_latitude = ?, current_longitude = ? WHERE id = ?`,
                [latitude, longitude, userId]
            );
        }
        // --- End block ---

        let updateData = {};
        const now = new Date();
        console.log("task status"+task.status);
        if (task.status === 'Open') {
            // Get previous task for this user (by checkin_at desc)
            const prevTask = await Task.getPreviousTask(userId, id);

            let distanceTravelled = 0;
            console.log("previous task"+JSON.stringify(prevTask));
            if (prevTask) {
                if (prevTask.tasktype_id === 1) {

                    // Fetch user's branch latitude/longitude
                    let branchRows = await db.query(
                        `SELECT b.branch_lat, b.branch_long 
                         FROM users u 
                         LEFT JOIN branches b ON u.branch_id = b.id 
                         WHERE u.id = ? LIMIT 1`, [userId]
                    );
                    console.log('userId:', userId);
                    let branch = branchRows && branchRows[0] ? branchRows[0] : null;
                    console.log('branchRows:', branchRows);
                    if (branchRows && branchRows[0]) {
                        branch = branchRows[0];
                    }

                    // Fetch user's home latitude/longitude
                    let userRows = await db.query(
                        `SELECT home_latitude, home_longitude, home_approval FROM users WHERE id = ? LIMIT 1`, [userId]
                    );
                    const userHome = userRows && userRows[0] ? userRows[0] : null;

                    // Calculate distance from branch to current, and home to current
                    let branchDistance = null, homeDistance = null, prevTaskDistance = null, prevTaskDistance1 = null;
                    if (branch && branch.branch_lat && branch.branch_long) {
                        branchDistance = await Task.calculateDistance(
                            branch.branch_lat,
                            branch.branch_long,
                            latitude,
                            longitude
                        );
                    }
                    if (prevTask && prevTask.checkin_lat && prevTask.checkin_long) {
                        prevTaskDistance = await Task.calculateDistance(
                            prevTask.checkin_lat,
                            prevTask.checkin_long,
                            branch.branch_lat,
                            branch.branch_long
                        );
                        prevTaskDistance1 = await Task.calculateDistance(
                            prevTask.checkin_lat,
                            prevTask.checkin_long,
                            latitude,
                            longitude
                        );
                    }
                    if ((userHome && userHome.home_latitude && userHome.home_longitude)||(userHome.home_approval == "Approved")) {
                        homeDistance = await Task.calculateDistance(
                            userHome.home_latitude,
                            userHome.home_longitude,
                            latitude,
                            longitude
                        );
                    }
                    console.log("branchDistance: " + branchDistance);
                    console.log("homeDistance: " + homeDistance);
                    // Save the shortest non-null distance
                    if(prevTaskDistance <= 0.5) {
                        prevTaskDistance1 = branchDistance;
                    }
                    else if (branchDistance !== null && homeDistance !== null) {
                        distanceTravelled = Math.min(branchDistance, homeDistance);
                    } else if (branchDistance !== null) {
                        distanceTravelled = branchDistance;
                    } else if (homeDistance !== null) {
                        distanceTravelled = homeDistance;
                    } else {
                        distanceTravelled = 0;
                    }
                } else {
                    // For other tasktype_id, use last task's checkin_lat/long
                    if (prevTask.checkin_lat && prevTask.checkin_long) {
                        distanceTravelled = await Task.calculateDistance(
                            prevTask.checkin_lat,
                            prevTask.checkin_long,
                            latitude,
                            longitude
                        );
                    }
                }
            }

            updateData = {
                status: 'Inprogress',
                checkin_lat: latitude,
                checkin_long: longitude,
                checkin_location: location,
                checkin_at: now,
                distance_travelled: distanceTravelled,
                ...(remarks && { remarks })
            };
        } 
        else if (task.status === 'Inprogress') {
            let timespent = null;
            if (task.checkin_at) {
                const checkinAt = new Date(task.checkin_at);
                timespent = Math.floor((now - checkinAt) / 1000);
            }
            
            let distanceTravelled=0;
            //let distanceTravelled = task.distance_travelled || 0;
            console.log(task.tasktype_id);
            
            // Fetch user's home lat/long and role info (moved outside conditional for all task types)
            console.log('userId:', userId);
            let userRows = await db.query(
                `SELECT u.home_latitude, u.home_longitude, u.roles_id, r.default_transport 
                 FROM users u 
                 LEFT JOIN roles r ON u.roles_id = r.id 
                 WHERE u.id = ? LIMIT 1`, [userId]
            );
            
            const userHome = userRows && userRows[0] ? userRows[0] : null;
            console.log('User data with role:', userHome);
            
            // Calculate distance for attendance task as before (Google API logic)
            if (task.tasktype_id === 1 && task.checkin_lat && task.checkin_long) {
                console.log("attendance");
                console.log(task.tasktype_id);
                const prevTask = await Task.getNextTask(userId, id);
                console.log("prevtask"+JSON.stringify(prevTask, null, 2));
                // Fetch user's branch lat/long
                let branchRows = await db.query(
                    `SELECT b.branch_lat, b.branch_long 
                     FROM users u 
                     LEFT JOIN branches b ON u.branch_id = b.id 
                     WHERE u.id = ? LIMIT 1`, [userId]
                );
                console.log(branchRows);
                const branch = branchRows && branchRows[0] ? branchRows[0] : null;
                // Prepare Google Maps Distance Matrix API call
                /*const origins = `${task.checkin_lat},${task.checkin_long}`;
                let destinations = [];
                if (branch && branch.branch_lat && branch.branch_long) {
                    destinations.push(`${branch.branch_lat},${branch.branch_long}`);
                }
                if (userHome && userHome.home_latitude && userHome.home_longitude) {
                    destinations.push(`${userHome.home_latitude},${userHome.home_longitude}`);
                }*/
                
                let branchDistance = null, homeDistance = null, prevTaskDistance = null, prevTaskDistance1 = null;
                if (branch && branch.branch_lat && branch.branch_long) {

                console.log(branch.branch_long);
                console.log("task checkin");
                console.log("task checkin"+prevTask.checkin_lat);
                     branchDistance = await Task.calculateDistance(
                        prevTask.checkin_lat,
                        prevTask.checkin_long,
                        branch.branch_lat,
                        branch.branch_long
                    );
                }
                if (prevTask && prevTask.checkin_lat && prevTask.checkin_long) {
                   /*prevTaskDistance = await Task.calculateDistance(
                        prevTask.checkin_lat,
                        prevTask.checkin_long,
                        branch.branch_lat,
                        branch.branch_long
                    );*/
                    prevTaskDistance1 = await Task.calculateDistance(
                        prevTask.checkin_lat,
                        prevTask.checkin_long,
                        latitude,
                        longitude
                    );
                }
                if (userHome && userHome.home_latitude && userHome.home_longitude) {
                    console.log(userHome.home_latitude);
                 homeDistance = await Task.calculateDistance(
                        prevTask.checkin_lat,
                        prevTask.checkin_long,
                        userHome.home_latitude,
                        userHome.home_longitude,
                    );
                }
                console.log("branchDistance: " + branchDistance);
                console.log("homeDistance: " + homeDistance);
                if(branchDistance <= 0.5) {
                    console.log("firstif: " + branchDistance);
                    distanceTravelled = branchDistance;
                }
                else if (branchDistance !== null && homeDistance !== null) {
                    
                    console.log("secondif: " + branchDistance);
                    distanceTravelled = Math.min(branchDistance, homeDistance);
                } else if (branchDistance !== null) {
                    console.log("thirdif: " + branchDistance);
                    distanceTravelled = branchDistance;
                } else if (homeDistance !== null) {
                    console.log("forthif: " + homeDistance);
                    distanceTravelled = homeDistance;
                } else {
                    console.log("fifthif: 0");
                    distanceTravelled = 0;
                }
                /*if (destinations.length > 0) {
                    const apiKey = process.env.GOOGLE_MAPS_API_KEY;
                    const url = `https://maps.googleapis.com/maps/api/distancematrix/json?origins=${origins}&destinations=${destinations.join('|')}&key=${apiKey}`;
                    const response = await axios.get(url);
                    const elements = response.data.rows[0].elements;

                    // Get the shortest distance in meters
                    let minDistance = null;
                    elements.forEach(el => {
                        if (el.status === 'OK') {
                            if (minDistance === null || el.distance.value < minDistance) {
                                minDistance = el.distance.value;
                            }
                        }
                    });
                    console.log(minDistance);
                    // Convert meters to kilometers
                    distanceTravelled = minDistance !== null ? (minDistance / 1000) : 0;
                }*/
                    console.log("tasktypeid distance: " + distanceTravelled);
            } else {
                // For other task types, fallback to haversine or your own method
                distanceTravelled = task.distance_travelled || 0;
            }
            
            console.log("outside distance: " + distanceTravelled);
            
            // Claim calculation logic
            let claimImageFileName = null;
            if (transport_mode === 'Public Transport' && claim_image) {
                claimImageFileName = claim_image.filename;
            }

            let conveyance_charge = 0;
            let finalClaimAmount = claim_amount || 0;
            
            // Determine transport mode - use provided transport_mode or default from role
            let effectiveTransportMode = transport_mode;
            if (!effectiveTransportMode && userHome && userHome.default_transport) {
                effectiveTransportMode = userHome.default_transport;
                console.log('Using default transport from role:', effectiveTransportMode);
            }
            // Calculate claim amount based on transport mode and distance
            if (effectiveTransportMode && effectiveTransportMode !== 'Public Transport') {
                // Fetch conveyance charge for the transport_mode
                let configRows = await db.query(
                    'SELECT conveyance_charge FROM conveyanceconfigs WHERE transport_mode = ? LIMIT 1',
                    [effectiveTransportMode]
                );
                conveyance_charge = configRows && configRows[0] ? configRows[0].conveyance_charge : 0;
                finalClaimAmount = conveyance_charge * distanceTravelled;
                console.log(`Claim calculation: ${conveyance_charge} × ${distanceTravelled} = ${finalClaimAmount}`);
            } else if (effectiveTransportMode === 'Public Transport') {
                // For public transport, claim amount is typically fixed or based on receipt
                finalClaimAmount = claim_amount || 0;
                console.log('Public transport claim amount:', finalClaimAmount);
            } else {
                // No transport mode specified, no claim
                finalClaimAmount = 0;
                console.log('No transport mode specified, claim amount: 0');
            }
            
            updateData = {
                status: 'Closed',
                checkout_lat: latitude,
                checkout_long: longitude,
                checkout_location: location,
                checkout_at: now,
                time_spent: timespent,
                distance_travelled: distanceTravelled,
                transport_mode: effectiveTransportMode  || null,
                claim_amount: finalClaimAmount || null,
                claim_remarks: claim_remarks || null,
                ack_contact_id: ack_contact_id || null,
                manual_checkout_at: actual_checkout_date || null,
                ...(remarks && { remarks })
            };
        } else {
            return res.json({
                code: 200,
                status: true,
                message: `Task status is already ${task.status}`,
                response: { status: task.status }
            });
        }
        console.log(updateData);
        await Task.updateTaskStatus(id, updateData);
        const taskdata = await Task.findById(id);

        res.json({
            code: 200,
            status: true,
            message: 'Task status updated successfully',
            response: {
                tasks: taskdata
            }
        });
        } catch (error) {
        console.error('Update task status error:', error);
        res.status(500).json({
            code: 500,
            status: false,
            message: 'Failed to update task status'
        });
    }
   
};
// In your controller
export const filterTasks = async (req, res) => {
    try {
        const userId = req.user.id;
        const { filter, filter_by } = req.body;
        const result = await Task.filterTasks(userId, filter, filter_by);

        res.json({
            code: 200,
            status: true,
            message: 'Tasks fetched successfully',
            response: {
                tasks: Array.isArray(result.tasks) ? result.tasks : [],
                isCheckinCheckout: typeof result.isCheckinCheckout === 'boolean' ? result.isCheckinCheckout : true,
                attendancecount: typeof result.attendancecount === 'number' ? result.attendancecount : 0,
                attendance: result.attendance ? result.attendance : null,
                
            }
        });
    } catch (error) {
        console.error('Filter tasks error:', error);
        res.status(500).json({
            code: 500,
            status: false,
            message: 'Failed to fetch tasks',
            response: {
                tasks: [],
                isCheckinCheckout: true,
                attendancecount: 0,
                atatendance:null
            }
        });
    }
};

export const attendanceHistory = async (req, res) => {
    try {
        const userId = req.user.id;
        const { filter, month, year } = req.query; // filter can be 'Today', 'Weekly', 'Monthly'

        let dateCondition = '';
        let params = [userId];

        if (filter === 'Today') {
            dateCondition = 'AND DATE(t.start_date) = CURDATE()';
        } else if (filter === 'Weekly') {
            dateCondition = 'AND YEARWEEK(t.start_date, 1) = YEARWEEK(CURDATE(), 1)';
        } else if (filter === 'Monthly') {
            // Use provided month/year or fallback to current
            const now = new Date();
            const monthVal = month ? parseInt(month, 10) : now.getMonth() + 1;
            const yearVal = year ? parseInt(year, 10) : now.getFullYear();
            dateCondition = 'AND YEAR(t.start_date) = ? AND MONTH(t.start_date) = ?';
            params.push(yearVal, monthVal);
        }

        const query = `
            SELECT 
                t.id, t.task, t.description, t.start_date, t.start_time, t.end_date, t.end_time,
                t.checkin_at, t.checkout_at, t.checkin_location, t.checkout_location, t.status, t.time_spent
            FROM tasks t
            WHERE t.user_id = ?
              AND t.tasktype_id = 1
              ${dateCondition}
            ORDER BY t.start_date DESC, t.start_time DESC
        `;

        // Properly destructure the result
        let attendanceRows = await db.query(query, params);

        res.json({
            code: 200,
            status: true,
            message: 'Attendance history fetched successfully',
            response: {
                attendance: Array.isArray(attendanceRows) ? attendanceRows : []
            }
        });
    } catch (error) {
        console.error('Attendance history error:', error);
        res.status(500).json({
            code: 500,
            status: false,
            message: 'Failed to fetch attendance history',
            response: {
                attendance: []
            }
        });
    }
};


export const conveyance = async (req, res) => {
    const response = {};
    try {
        const { filter, month, year } = req.body;
        if (!filter) {
            return res.status(200).json({
                code: 400,
                status: false,
                message: 'Filter field is required',
                response
            });
        }

        const userId = req.user.id;
        let tasks = [];
        let claimapply = false;
        let params = [userId];
        let dateCondition = '';
        let groupBy = 'GROUP BY t.start_date';

        // 1. Filter logic
        if (filter === 'Today') {
            dateCondition = 'AND t.start_date = CURDATE()';
        } else if (filter === 'Weekly') {
            dateCondition = 'AND t.start_date BETWEEN DATE_SUB(CURDATE(), INTERVAL 7 DAY) AND CURDATE()';
        } else if (filter === '1-15 days') {
            dateCondition = 'AND t.start_date BETWEEN DATE_FORMAT(CURDATE(), "%Y-%m-01") AND DATE_FORMAT(CURDATE(), "%Y-%m-15")';
        } else if (filter === '16-31 days') {
            dateCondition = 'AND t.start_date BETWEEN DATE_FORMAT(DATE_SUB(CURDATE(), INTERVAL 1 MONTH), "%Y-%m-16") AND DATE_FORMAT(DATE_SUB(CURDATE(), INTERVAL 1 MONTH), "%Y-%m-31")';
        } else if (filter === 'Monthly') {
            // Use provided month/year or fallback to current
            const now = new Date();
            const monthVal = month ? parseInt(month, 10) : now.getMonth() + 1;
            const yearVal = year ? parseInt(year, 10) : now.getFullYear();
            dateCondition = 'AND MONTH(t.checkin_at) = ? AND YEAR(t.checkin_at) = ?';
            params.push(monthVal, yearVal);
        } else {
            return res.status(200).json({
                code: 400,
                status: false,
                message: 'Invalid filter value',
                response
            });
        }

        // 2. Get conveyance config
        const [conveyanceConfigRows] = await db.query('SELECT * FROM conveyanceconfigs LIMIT 1');
        const conveyance_charge = conveyanceConfigRows?.[0]?.conveyance_charge || 0;

        // 3. Get attendance config (for deviation logic)
        const [attendanceConfigRows] = await db.query('SELECT * FROM attendanceconfigs LIMIT 1');
        const attendanceConfig = attendanceConfigRows?.[0] || {};

        // 4. Get grouped tasks by start_date
        const tasksQuery = `
            SELECT t.start_date
            FROM tasks t
            LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
            WHERE t.user_id = ?
              AND t.status = 'Closed'
              AND tt.allow_checkin_checkout = 1
              ${dateCondition}
            ${groupBy}
        `;
        let taskRows = await db.query(tasksQuery, params);

        if (!taskRows || taskRows.length === 0) {
            return res.status(200).json({
                code: 204,
                status: false,
                message: 'Conveyance info not exists',
                response
            });
        }

        // 5. Check claimapply for 1-15 and 16-31 days
        if (filter === '1-15 days' || filter === '16-31 days') {
            const claimCountQuery = `
                SELECT COUNT(*) as claimcount
                FROM tasks t
                WHERE t.user_id = ?
                  AND t.status = 'Closed'
                  AND t.claim_status != ''
                  ${dateCondition}
                ${groupBy}
            `;
            let claimCountRows = await db.query(claimCountQuery, [userId]);
            if (claimCountRows && claimCountRows[0]?.claimcount > 0) {
                claimapply = true;
            }
        }

        // 6. For each date, calculate distance, amount, timespent
        const conveyances = [];
        for (const task of taskRows) {
            const date = task.start_date;

            // Deviation logic
            let deviationsids = [];
            if (attendanceConfig.allow_outsideapproval === 1) {
                // Find deviation check
                let deviationCheckRows = await db.query(
                    `SELECT * FROM tasks WHERE user_id = ? AND DATE(checkin_at) = ? AND tasktype_id = 1 AND deviation_approval_status = 'Pending' LIMIT 1`,
                    [userId, date]
                );
                const deviationCheck = deviationCheckRows?.[0];
                if (deviationCheck) {
                    if (deviationCheck.deviation_in) {
                        let firstVisitRows = await db.query(
                            `SELECT id FROM tasks WHERE user_id = ? AND DATE(checkin_at) = ? AND tasktype_id != 1 ORDER BY checkin_at ASC LIMIT 1`,
                            [userId, date]
                        );
                        if (firstVisitRows?.[0]) deviationsids.push(firstVisitRows[0].id);
                    }
                    if (deviationCheck.deviation_out) {
                        let lastVisitRows = await db.query(
                            `SELECT id FROM tasks WHERE user_id = ? AND DATE(checkin_at) = ? AND tasktype_id = 1 AND status = 'Closed' LIMIT 1`,
                            [userId, date]
                        );
                        if (lastVisitRows?.[0]) deviationsids.push(lastVisitRows[0].id);
                    }
                }
            }

            // Build exclusion for deviations
            let notInClause = '';
            if (deviationsids.length > 0) {
                notInClause = `AND t.id NOT IN (${deviationsids.map(() => '?').join(',')})`;
            }

            // Calculate distance
            let distanceRows = await db.query(
                `SELECT SUM(t.distance_travelled) as distance
                 FROM tasks t
                 LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
                 WHERE t.user_id = ?
                   AND DATE(t.checkin_at) = ?
                   AND t.status = 'Closed'
                   AND tt.allow_checkin_checkout = 1
                   ${notInClause}`,
                [userId, date, ...deviationsids]
            );
            const distance = distanceRows?.[0]?.distance || 0;

            // Calculate claim amount
            let claimAmountRows = await db.query(
                `SELECT SUM(t.claim_amount) as amount
                 FROM tasks t
                 LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
                 WHERE t.user_id = ?
                   AND DATE(t.checkin_at) = ?
                   AND t.status = 'Closed'
                   AND tt.allow_checkin_checkout = 1
                   ${notInClause}`,
                [userId, date, ...deviationsids]
            );
            const amount = claimAmountRows?.[0]?.amount || 0;

            // Calculate timespent
           let timespentRows = await db.query(
                `SELECT SUM(t.time_spent) as timespent
                 FROM tasks t
                 LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
                 WHERE t.user_id = ?
                   AND DATE(t.checkin_at) = ?
                   AND t.status = 'Closed'
                   AND tt.allow_checkin_checkout = 1
                   ${notInClause}`,
                [userId, date, ...deviationsids]
            );
            const timespent = timespentRows?.[0]?.timespent || 0;

            // Calculate charge
            const charge = conveyance_charge * distance;

            conveyances.push({
                conveyance_date: date,
                distance,
                amount,
                timespent
            });
        }

        response.conveyance = conveyances;
        response.claimapply = claimapply;

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Conveyance info',
            response
        });

    } catch (error) {
        console.error('Conveyance error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'Conveyance - ' + error.message,
            response
        });
    }
};

export const viewConveyance = async (req, res) => {
    const response = {};
    try {
        const { date } = req.body;
        if (!date) {
            return res.status(200).json({
                code: 400,
                status: false,
                message: 'Conveyance date field is required',
                response
            });
        }

        const userId = req.user.id;
        const userRoleId = req.user.roles_id;

        const result = await Task.viewConveyance(userId, userRoleId, date);
        console.log(result);
        if (result.conveyance && result.conveyance.length > 0) {
            return res.status(200).json({
                code: 200,
                status: true,
                message: 'Conveyance info',
                response: {
                    conveyance: result.conveyance,
                    transport_mode_edit_bike: result.transport_mode_edit_bike
                }
            });
        } else {
            return res.status(200).json({
                code: 204,
                status: false,
                message: 'Conveyance info not exists',
                response
            });
        }
    } catch (error) {
        console.error('View Conveyance error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'View Conveyance - ' + error.message,
            response
        });
    }
};

export const addClaim = async (req, res) => {
    const response = {};
    try {
        const { task_id, transport_mode, amount, remarks } = req.body;
        if (!task_id) {
            return res.status(200).json({
                code: 200,
                status: false,
                message: 'Task ID field is required',
                response
            });
        }

        const files = req.files || [];
        const user_id = req.user.id;

        const result = await Task.applyClaim({
            task_id,
            transport_mode,
            amount,
            remarks,
            files,
            user_id
        });

        if (result.error) {
            return res.status(200).json({
                code: 204,
                status: false,
                message: result.error,
                response: null
            });
        }

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Claim added successfully',
            response: result
        });
    } catch (error) {
        console.error('Add claim error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'Error - ' + error.message,
            response
        });
    }
};

export const readdClaim = async (req, res) => {
    const response = {};
    try {
        const { task_id, transport_mode, amount, remarks } = req.body;
        if (!task_id) {
            return res.status(200).json({
                code: 200,
                status: false,
                message: 'Task ID field is required',
                response
            });
        }

        const files = req.files || [];
        const user_id = req.user.id;

        const result = await Task.reapplyClaim({
            task_id,
            transport_mode,
            amount,
            remarks,
            files,
            user_id
        });

        if (result.error) {
            return res.status(200).json({
                code: 204,
                status: false,
                message: result.error,
                response: null
            });
        }

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Claim added successfully',
            response: result
        });
    } catch (error) {
        console.error('Readd claim error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'Error - ' + error.message,
            response
        });
    }
};

export const modeOfTransport = async (req, res) => {
    try {
        const user = req.user;
        console.log(user);
        const userId = user && user.id ? user.id : null;
        const userRoleId = user && user.role ? user.role : null;

        if (!userId || !userRoleId) {
            return res.status(200).json({
                code: 400,
                status: false,
                message: 'User or role not found',
                response: {}
            });
        }

        const result = await Task.getModeOfTransport(userId, userRoleId);

        if (result.notFound) {
            return res.status(200).json({
                code: 204,
                status: false,
                message: 'Transport Info not found',
                response: {}
            });
        }

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Transport info',
            response: { modeoftransport: result.modeoftransport }
        });
    } catch (error) {
        console.error('Task info error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'Task info -' + error.message,
            response: {}
        });
    }
};

export const applyClaimList = async (req, res) => {
    const response = {};
    try {
        const { from_date, to_date } = req.body;
        if (!from_date || !to_date) {
            return res.status(200).json({
                code: 400,
                status: false,
                message: 'From and To Date fields are required',
                response:[]
            });
        }

        const userId = req.user.id;
        const result = await Task.getApplyClaimList(userId, from_date, to_date);

        if (result.error) {
            return res.status(200).json({
                code: 400,
                status: false,
                message: result.error,
                response:[]
            });
        }

        if (result.notExists) {
            return res.status(200).json({
                code: 204,
                status: false,
                message: 'Apply claim info not exists',
                response:[]
            });
        }

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Apply claim info',
            response: result
        });
    } catch (error) {
        console.error('Apply claim error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'Apply claim -' + error.message,
            response:[]
        });
    }
};

export const applyClaim = async (req, res) => {
    const response = {};
    try {
        const { from_date, to_date } = req.body;
        if (!from_date || !to_date) {
            return res.status(200).json({
                code: 400,
                status: false,
                message: 'From and To Date fields are required',
                response
            });
        }

        const userId = req.user.id;
        const result = await Task.applyClaimBulk(userId, from_date, to_date);

        if (result.notExists) {
            return res.status(200).json({
                code: 204,
                status: false,
                message: 'Apply claim failure',
                response
            });
        }

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Apply claim successfull',
            response: { tasks: result.tasks }
        });
    } catch (error) {
        console.error('Apply claim error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'Apply claim -' + error.message,
            response
        });
    }
};

export const claimHistory = async (req, res) => {
    const response = {};
    try {
        const userId = req.user.id;
        const { from_date, to_date } = req.body;

        const result = await Task.getClaimHistory(userId, from_date, to_date);

        if (result) {
            return res.status(200).json({
                code: 200,
                status: true,
                message: 'Claim history info',
                response: result
            });
        } else {
            return res.status(200).json({
                code: 204,
                status: false,
                message: 'Claim history info not exists',
                response
            });
        }
    } catch (error) {
        console.error('Claim history error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'Login -' + error.message,
            response
        });
    }
};


export const addTaskNote = async (req, res) => {
    try {
        const user_id = req.user.id;
        const { project_id, task_id } = req.body;
        const notes = typeof req.body.notes === 'undefined' || req.body.notes === null ? '' : req.body.notes;
        // Support both single and multiple files
        const file = req.file || (req.files && req.files.file ? req.files.file[0] : null);

        // Make either notes or file required (at least one must be present)
        if (!project_id || !task_id) {
            return res.status(200).json({
                code: 400,
                status: false,
                message: 'project_id, task_id are required',
                response: {}
            });
        }
        if ((!notes || notes.trim() === '') && !file) {
            return res.status(200).json({
                code: 400,
                status: false,
                message: 'Either notes or file is required',
                response: {}
            });
        }

        const result = await Task.addTaskNote({
            project_id,
            task_id,
            user_id,
            notes,
            file
        });

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Task note added successfully',
            response: result
        });
    } catch (error) {
        console.error('Add task note error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'Add task note - ' + error.message,
            response: {}
        });
    }
};

function formatDate(date) {
    if (!date) return null;
    return new Date(date).toLocaleDateString('en-GB', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric'
    });
}


export const viewTaskDetails = async (req, res) => {
    let response = {};
    try {
        const { id } =  req.params;
        const task_id=id;
        if (!task_id) {
            return res.status(200).json({
                code: 400,
                status: false,
                message: 'Task ID field is required',
                response: {}
            });
        }

        const userId = req.user.id;
        const date = new Date().toISOString().slice(0, 10);

        // Get user, role, branch
        let userRows = await db.query(`SELECT * FROM users WHERE id = ? LIMIT 1`, [userId]);
        const user = userRows[0];
        let roleRows = await db.query(`SELECT * FROM roles WHERE id = ? LIMIT 1`, [user.roles_id]);
        const role = roleRows[0];
        let branchRows = await db.query(`SELECT * FROM branches WHERE id = ? LIMIT 1`, [user.branch_id]);
        const branch = branchRows[0];

        // Attendance config
        let attendanceConfigRows = await db.query(`SELECT * FROM attendanceconfigs LIMIT 1`);
        const attendanceconfig = attendanceConfigRows[0];

        // Open task (site visit or similar)
        let openTaskRows = await db.query(
            `SELECT * FROM tasks WHERE user_id = ? AND tasktype_id IN (3,8) AND status = 'Inprogress' LIMIT 1`,
            [userId]
        );
        const opentask = openTaskRows[0] || null;

        // isCheckinCheckout logic
        let isCheckinCheckout = true;
        if (opentask) {
            isCheckinCheckout = !opentask.checkout_at ? false : false;
        } else {
            isCheckinCheckout = true;
        }

        // Attendance check
        let attendanceCheckRows = await db.query(
            `SELECT * FROM tasks WHERE user_id = ? AND tasktype_id = 1 AND start_date = ? AND checkin_at IS NOT NULL LIMIT 1`,
            [userId, date]
        );
        const attendancecheck = attendanceCheckRows[0] || null;
        let attendancecount = 0;
        if (attendancecheck) {
            attendancecount = !attendancecheck.checkout_at ? 1 : 2;
        }

        // Main task with relations
        let taskRows = await db.query(
            `SELECT * FROM tasks WHERE id = ? LIMIT 1`, [task_id]
        );
        const task = taskRows[0] || null;
        if (task && task.ack_contact_id) {
            let ackContactRows = await db.query(
                `SELECT id, contact_name, email, mobile_no, company_id FROM contacts WHERE id = ? LIMIT 1`,
                [task.ack_contact_id]
            );
            task.ack_contact = ackContactRows && ackContactRows[0] ? ackContactRows[0] : null;
        }
        if (task) {
            // Fetch and add tasktypename
            let tasktypeRow = await db.query(
                `SELECT activity_name FROM tasktypes WHERE id = ? LIMIT 1`, [task.tasktype_id]
            );
            task.tasktypename = tasktypeRow && tasktypeRow[0] ? tasktypeRow[0].activity_name : null;

            // Fetch and add tasksubcategory_name
            let subcategoryRow = await db.query(
                `SELECT subcategory FROM tasksubcategories WHERE id = ? LIMIT 1`, [task.tasksubcategory_id]
            );
            task.tasksubcategory_name = subcategoryRow && subcategoryRow[0] ? subcategoryRow[0].subcategory : null;

            // Fetch and add product_name
            let productRow = await db.query(
                `SELECT name FROM products WHERE id = ? LIMIT 1`, [task.product_id]
            );
            task.product_name = productRow && productRow[0] ? productRow[0].name : null;

            // Fetch and add taskcategoryoption_name
            let optionRow = await db.query(
                `SELECT dropdown FROM taskcategoryoptions WHERE id = ? LIMIT 1`, [task.taskcategoryoption_id]
            );
            task.taskcategoryoption_name = optionRow && optionRow[0] ? optionRow[0].dropdown : null;
        }

        // Notes for this task
        let notesRows = await db.query(
            `SELECT * FROM tasknotes WHERE task_id = ?`, [task_id]
        );
        const notes = notesRows || [];

        // Task type with questions
        let tasktypes = null;
        if (task) {
            let tasktypeRows = await db.query(
                `SELECT * FROM tasktypes WHERE id = ? AND status = 1 LIMIT 1`, [task.tasktype_id]
            );
            tasktypes = tasktypeRows[0] || null;
           
        }

       

        // Files (taskuploads with taskactions)
        let filesRows = await db.query(`
             SELECT tu.id as upload_id, tu.task_id, tu.file_name as file_name, tu.created_at, tu.updated_at
            FROM taskuploads tu
            WHERE tu.task_id = ?
            ORDER BY tu.created_at DESC`, [task_id]
        );
        const files = [];

        for (const file of filesRows) {
            // Fetch all taskactions for this upload
            let actionsRows = await db.query(
                `SELECT id, remarks, action_id, action_name, created_at, updated_at
                 FROM taskactions
                 WHERE taskupload_id = ?`,
                [file.upload_id]
            );
            file.taskactions = actionsRows || [];
            files.push(file);
        }

        // Eligible transport modes
        const eligibletransport = role?.eligible_transport || '';
        const modes = eligibletransport ? eligibletransport.split(',') : [];

        // Transport mode
        const transportmode = role?.vehicle || null;

        // --- Add project, company, and contact details ---
        let project = null, company = null, contact = null, assigned_users=null;
        if (task) {
            // Project details
            let projectRows = await db.query(
                `SELECT * FROM projects WHERE id = ? LIMIT 1`, [task.project_id]
            );
            project = projectRows[0] || null;

            // Company details (if project exists)
            if (project && project.customer_id) {
                let companyRows = await db.query(
                    `SELECT * FROM companies WHERE id = ? LIMIT 1`, [project.customer_id]
                );
                company = companyRows[0] || null;
            }
           
               
                // Fetch assigned users from projectassignees table
               const assigneesQuery = `
            SELECT pa.user_id, u.name, u.emp_id, r.name as role_name
            FROM projectassignees pa
            LEFT JOIN users u ON pa.user_id = u.id
            LEFT JOIN roles r ON u.roles_id = r.id
            WHERE pa.project_id = ? AND pa.status = 1
        `;
        let assigneesResult = await db.query(assigneesQuery, [task.project_id]);

               
         assigned_users = assigneesResult || null;
        
           
        
            // Contact details (if task has contact_id)
            if (task.contact_id) {
                let contactRows = await db.query(
                    `SELECT * FROM contacts WHERE id = ? LIMIT 1`, [task.contact_id]
                );
                contact = contactRows[0] || null;
            }
        }

        if (task.start_date) {
            task.start_date =  formatDate(task.start_date).replaceAll('/','-').split('-').reverse().join('-')
        }
        
        if (task) {
            
            response = {
                task,
                project,    // <-- added
                company,    // <-- added
                contact,    // <-- added
                notes,
                tasktypes,
                assigned_users,
                files,
                attendancecount,
                isCheckinCheckout,
                attendanceconfig,
                branch,
                transportmode,
                transportmodelist: modes
            };
            return res.status(200).json({
                code: 200,
                status: true,
                message: 'Task info',
                response
            });
        } else {
            return res.status(200).json({
                code: 204,
                status: false,
                message: 'Task info not exists',
                response: {}
            });
        }
    } catch (error) {
        console.error('View task details error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'Login -' + error.message,
            response: {}
        });
    }
};

export const taskActionList = async (req, res) => {
    const response = {};
    try {
        const { task_id } = req.body;
        if (!task_id) {
            return res.status(400).json({
                code: 400,
                status: false,
                message: 'Task ID field is required',
                response
            });
        }

        const userId = req.user.id;
        // Get user and role
        let userRows = await db.query('SELECT * FROM users WHERE id = ? LIMIT 1', [userId]);
        const user = userRows[0];
        if (!user) {
            return res.status(404).json({
                code: 404,
                status: false,
                message: 'User not found',
                response
            });
        }
        let roleRows = await db.query('SELECT * FROM roles WHERE id = ? LIMIT 1', [user.roles_id]);
        const role = roleRows[0];

        // Get task with related data
        let taskRows = await db.query(
            `SELECT t.*, 
                    tt.activity_name as tasktype_name, 
                    c.company_name, 
                    ct.contact_name
             FROM tasks t
             LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
             LEFT JOIN companies c ON t.company_id = c.id
             LEFT JOIN contacts ct ON t.contact_id = ct.id
             WHERE t.id = ? LIMIT 1`, [task_id]
        );
        const task = taskRows[0];
        if (!task) {
            return res.status(404).json({
                code: 404,
                status: false,
                message: 'Task not found',
                response
            });
        }

        // Action lists for this task type
        let actionlists = await db.query(
            'SELECT * FROM taskactionlists WHERE task_type_id = ? AND status = 1',
            [task.tasktype_id]
        );

        // Task type details
        let tasktypeRows = await db.query(
            'SELECT * FROM tasktypes WHERE id = ? AND status = 1 LIMIT 1',
            [task.tasktype_id]
        );
        const tasktypes = tasktypeRows[0] || null;

        // Questions for this task type
        let project = await db.query(
            'SELECT * FROM projects WHERE id = ?',
            [task.project_id]
        );

        // Eligible transport
        const eligibleTransport = role?.eligible_transport || '';
        const transportmodelist = eligibleTransport ? eligibleTransport.split(',').map(s => s.trim()) : [];

        // All transport configs
        let transport = await db.query('SELECT * FROM conveyanceconfigs');

        // Ticket status except id 5
        let projectstatus = project[0]?.status || 'Pending';

        // Contact list for company or ticket contact
        let contactlist = await db.query(
            'SELECT * FROM contacts WHERE company_id = ? OR id = ?',
            [task.company_id, task.contact_id]
        );

        // Open task count for this user and ticket
        let opentaskcount = 0;
        if (task.ticket_id) {
            let opentaskcountRows = await db.query(
                `SELECT COUNT(*) as count FROM tasks 
                 WHERE user_id = ? AND project_id = ? AND tasktype_id != 6 AND status IN ('Open','Inprogress')`,
                [userId, task.project_id]
            );
            opentaskcount = opentaskcountRows[0]?.count || 0;
        }

        response.actionlists = actionlists;
        response.tasktypes = tasktypes;
        response.task = task;
        response.transportmode = role?.vehicle || null;
        response.transportmodelist = transportmodelist;
        response.currentticketstatus = projectstatus;
        response.contacts = contactlist;
        response.opentaskcount = opentaskcount;

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Task info',
            response
        });

    } catch (e) {
        return res.status(500).json({
            code: 500,
            status: false,
            message: 'Error - ' + e.message,
            response
        });
    }
};
export const aiTaskFromVoicenote = async (req, res) => {
    try {
        if (!req.files || !req.files.voicenote || !req.files.voicenote[0]) {
            return res.status(400).json({ status: false, message: 'No voicenote uploaded' });
        }
        const voicenotePath = req.files.voicenote[0].path;

        // 1. Transcribe audio to text
        const transcript = await transcribeAudio(voicenotePath);
        console.log('Transcription:', transcript);
        // 2. Extract task details using AI/NLP
        const {
            project_name,
            task_type,
            assigned_engineer,
            task_date,
            task_time
        } = await extractTaskDetails(transcript);

        // --- Normalize date ---
        let normalizedDate = task_date;
        try {
            let cleaned = normalizedDate.replace(/(\d+)(st|nd|rd|th)/i, '$1');
            let parsed = parse(cleaned, 'd MMMM', new Date());
            if (isNaN(parsed)) {
                parsed = parse(cleaned + ' ' + new Date().getFullYear(), 'd MMMM yyyy', new Date());
            }
            if (!isNaN(parsed)) {
                normalizedDate = format(parsed, 'yyyy-MM-dd');
            }
        } catch (e) {
            // fallback: keep as is
        }

        // 3. Find related IDs in DB
        let projectRows = await db.query('SELECT id FROM projects WHERE project_name = ? LIMIT 1', [project_name]);
        const project = projectRows[0] || null;

        let engineerRows = await db.query('SELECT id FROM users WHERE name = ? LIMIT 1', [assigned_engineer]);
        const engineer = engineerRows[0] || null;

        let tasktypeRows = await db.query('SELECT id FROM tasktypes WHERE activity_name = ? LIMIT 1', [task_type]);
        const tasktype = tasktypeRows[0] || null;

        if (!project || !engineer || !tasktype) {
            return res.status(400).json({ status: false, message: 'Could not match all extracted fields to database records.' });
        }

        // 4. Create the task
        let result = await db.query(
            `INSERT INTO tasks (task,project_id, tasktype_id, user_id, start_date, start_time, created_at, updated_at,created_date,status)
             VALUES (?,?, ?, ?, ?, ?, NOW(), NOW(),NOW(),'Open')`,
            [task_type, project.id, tasktype.id, engineer.id, normalizedDate, task_time]
        );

        res.json({
            status: true,
            message: 'Task created from voicenote',
            task_id: result.insertId,
            extracted: { project_name, task_type, assigned_engineer, task_date, task_time }
        });
    } catch (error) {
        res.status(500).json({ status: false, message: error.message });
    }
};
export const extractTextFromVoicenote = async (req, res) => {
    try {
        if (!req.files || !req.files.voicenote || !req.files.voicenote[0]) {
            return res.status(400).json({ status: false, message: 'No voicenote uploaded' });
        }
        const voicenotePath = req.files.voicenote[0].path;
        const transcript = await transcribeAudio(voicenotePath);
        res.json({ status: true, transcript });
    } catch (error) {
        res.status(500).json({ status: false, message: error.message });
    }
};