import { findAllTasks, findTaskById } from "../models/taskmodel.js";
import { Task } from '../models/task.js';

async function getTasks(req, res) {
    try {
        const { start_date, end_date } = req.query;
        
        // Validate date format if provided
        if (start_date && !isValidDate(start_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid start_date format. Use YYYY-MM-DD"
            });
        }
        
        if (end_date && !isValidDate(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid end_date format. Use YYYY-MM-DD"
            });
        }

        const tasks = await findAllTasks(start_date, end_date);
        res.status(200).json(
            {
                status: 200,
                success: true,
                message: "Tasks list fetched successfully",
                data: tasks
            }
        );
    } catch (error) {
        res.status(500).json(
            {   
                status: 500,
                success: false,
                message: "Error fetching tasks",
                error: error.message
            }
        );
    }
}

// Helper function to validate date format
function isValidDate(dateString) {
    const regex = /^\d{4}-\d{2}-\d{2}$/;
    if (!regex.test(dateString)) return false;
    
    const date = new Date(dateString);
    return date instanceof Date && !isNaN(date) && date.toISOString().slice(0, 10) === dateString;
}

async function getTaskById(req, res) {
    const { id } = req.params;
    try {
        const task = await findTaskById(id);
        res.status(200).json(
            {
                status: 200,
                success: true,
                message: "Task details fetched successfully",
                data: task
            }
        );
    } catch (error) {
        res.status(500).json(
            {
                status: 500,
                success: false,
                message: "Error fetching task",
                error: error.message
            }
        );
    }
}

const createTask = async (req, res) => {
    try {
        const userId = req.user.id;
        const taskData = req.body;

        const taskId = await Task.create(taskData, userId);
        const task = await Task.findById(taskId);

        res.json({
            code: 200,
            status: true,
            message: 'Task created successfully',
            response: task
        });
    } catch (error) {
        console.error('Create task error:', error);
        res.json({
            code: 500,
            status: false,
            message: error.message || 'Failed to create task'
        });
    }
};

const updateTask = async (req, res) => {
    try {
        const { id } = req.params;
        const taskData = req.body;

        await Task.update(id, taskData);
        const task = await Task.findById(id);

        res.json({
            code: 200,
            status: true,
            message: 'Task updated successfully',
            response: task
        });
    } catch (error) {
        console.error('Update task error:', error);
        res.json({
            code: 500,
            status: false,
            message: 'Failed to update task'
        });
    }
};

export {      
    getTasks,
    getTaskById,
    createTask,
    updateTask
}