import { User } from '../models/user.js';
import { hashPassword } from '../utils/hash.js';
import { db } from '../utils/database.js';
export const createUser = async (req, res) => {
    try {
        // Hash password before storing
        const hashedPassword = hashPassword(req.body.password);
        const userData = {
            ...req.body,
            password: hashedPassword
        };

        const userId = await User.create(userData, req.user.id);
        const user = await User.findById(userId);
        
        res.status(201).json({
            status: 201,
            message: 'User created successfully',
            user
        });
    } catch (error) {
        console.error('Create user error:', error);
        
        if (error.message === 'Mobile number already exists') {
            return res.status(409).json({
                status: 409,
                message: 'Mobile number already exists'
            });
        }

        res.status(500).json({
            status: 500,
            message: 'Failed to create user'
        });
    }
};

export const updateUser = async (req, res) => {
    try {
        const { id } = req.params;
        await User.update(id, req.body);
        const user = await User.findById(id);

        res.json({
            status: 200,
            message: 'User updated successfully',
            user
        });
    } catch (error) {
        console.error('Update user error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to update user'
        });
    }
};

export const deleteUser = async (req, res) => {
    try {
        const { id } = req.params;
        await User.softDelete(id);

        res.json({
            status: 200,
            message: 'User deleted successfully'
        });
    } catch (error) {
        console.error('Delete user error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to delete user'
        });
    }
};

export const listUsers = async (req, res) => {
    try {
        const users = await User.list();
        
        res.json({
            status: 200,
            message: 'Users retrieved successfully',
            users
        });
    } catch (error) {
        console.error('List users error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to retrieve users'
        });
    }
};

export const datatableusers = async (req, res) => {
    try {
        // Get query params with defaults
        const {
            start = 1,
            length = 10,
            order_by = 'created_at',
            order_dir = 'DESC',
            search = ''
        } = req.query;

        const page = parseInt(start);
        const limit = parseInt(length);
        const sortBy = order_by;
        const sortOrder = order_dir;

        const { users, total } = await User.datatableList({ page, limit, sortBy, sortOrder, search });
        const totalPages = Math.ceil(total / limit);
        const paginationMeta = {
            total,
            page,
            limit,
            totalPages
        };
        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Users list fetched successfully',
            data: users,
            total,
            pagination: paginationMeta
        });
    } catch (error) {
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Error fetching datatable users',
            error: error.message
        });
    }
};

export const getUsersByRole = async (req, res) => {
    try {
        const { roleId } = req.params;

        if (!roleId) {
            return res.status(400).json({
                status: 400,
                message: 'Role ID is required'
            });
        }

        const users = await User.getUsersByRole(roleId);

        res.json({
            status: 200,
            message: 'Users retrieved successfully',
            users
        });
    } catch (error) {
        console.error('Get users by role error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to retrieve users'
        });
    }
};

export const changePassword = async (req, res) => {
    try {
        const userId = req.user.id;
        const { new_password, confirm_password } = req.body;

        if (!new_password || !confirm_password) {
            return res.status(200).json({
                code: 400,
                status: false,
                message: 'Old and new password are required',
                response: {}
            });
        }

        const result = await User.changePassword(userId, new_password, confirm_password);

        if (result.error) {
            return res.status(200).json({
                code: 400,
                status: false,
                message: result.error,
                response: {}
            });
        }

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Password changed successfully',
            response: {}
        });
    } catch (error) {
        console.error('Change password error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'Change password - ' + error.message,
            response: {}
        });
    }
};

export const logout = async (req, res) => {
    try {
        const userId = req.user.id;

        // Clear device_token and device_id on logout
        await db.query(
            `UPDATE users SET device_token = NULL, device_id = NULL, updated_at = NOW() WHERE id = ?`,
            [userId]
        );

        return res.status(200).json({
            code: 200,
            status: true,
            message: 'Logout successful',
            response: {}
        });
    } catch (error) {
        console.error('Logout error:', error);
        return res.status(200).json({
            code: 500,
            status: false,
            message: 'Logout failed',
            response: {}
        });
    }
};

export const updateHomeLocation = async (req, res) => {
    try {
        const userId = req.user.id; // or req.params.id if admin updates
        const { home_latitude, home_longitude, home_address } = req.body;
console.log('Updating home location for user:', userId, home_latitude, home_longitude);
        if (!home_latitude || !home_longitude) {
            return res.status(400).json({
                status: 400,
                message: 'home_latitude and home_longitude are required'
            });
        }

        await User.updateHomeLocation(userId, home_latitude, home_longitude, home_address);

        res.status(200).json({
            status: 200,
            message: 'Home location updated and approval set to Pending'
        });
    } catch (error) {
        console.error('Update home location error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to update home location'
        });
    }
};

export const getAssignableUsers = async (req, res) => {
    try {
        const { project_id } = req.query; // or req.body, as per your API design
        const loggedInUserId = req.user.id; // assuming you set this in your auth middleware

        // Get logged-in user details
        const loggedInUser = await User.findById(loggedInUserId);
        if (!loggedInUser) {
            return res.status(404).json({ status: false, message: 'Logged-in user not found' });
        }
        // Fetch access_scope from roles table
        let roleRow = await db.query(
            'SELECT scope FROM roles WHERE id = ? LIMIT 1',
            [loggedInUser.roles_id]
        );
        const accessScope = roleRow[0] ? roleRow[0].scope : null;

        let users = [];
        console.log('Project ID:', project_id); 
        console.log('Access scope:', accessScope);
        if (accessScope === 'Global') {
            // Fetch all users except those already assigned to the project
            users = await db.query(`
                SELECT u.*


                FROM users u
                WHERE u.active = 1
                  AND u.id NOT IN (
                      SELECT user_id FROM projectassignees WHERE project_id = ? and status = 1
                  )
                ORDER BY u.created_at DESC
            `, [project_id]);
        } else if (accessScope === 'Branch') {
            // Get branch IDs as array
            const branchIds = loggedInUser.branch_id
                ? (Array.isArray(loggedInUser.branch_id)
                    ? loggedInUser.branch_id
                    : loggedInUser.branch_id.split(',').map(id => id.trim()))
                : [];

            if (branchIds.length === 0) {
                return res.json({ status: true, users: [] });
            }
            console.log("loginid"+loggedInUserId);
            console.log("branchid"+branchIds);
            console.log("project"+project_id);
            // Fetch users in the same branches, not assigned to the project
            users = await db.query(`
                SELECT u.*
                FROM users u
                WHERE u.reporting_to=? and u.active = 1
                  AND (
                      ${branchIds.map(() => `FIND_IN_SET(?, u.branch_id)`).join(' OR ')}
                  )
                  AND u.id NOT IN (
                      SELECT user_id FROM projectassignees WHERE project_id = ? and status = 1
                  )
                ORDER BY u.created_at DESC
            `, [loggedInUserId,...branchIds, project_id]);
        } else {
            // Other access scopes (if any)
            return res.json({ status: true, users: [] });
        }

        res.json({ status: true, users });
    } catch (error) {
        res.status(500).json({ status: false, message: error.message });
    }
};