import { db } from '../utils/database.js';

export async function findAllActionReports({
    userId,
    branch,
    userIds,
    startDate,
    endDate
}) {
    try {
        // Step 1: Get User Role & Branch
        let userRows = await db.query(
            'SELECT id, roles_id, branch_id FROM users WHERE id = ?',
            [userId]
        );
        if (!userRows.length) throw new Error(`User not found with id: ${userId}`);
        const user = userRows[0];

        // Step 2: Get Role Access Scope
        let roleRows = await db.query(
            'SELECT access_scope FROM roles WHERE id = ?',
            [user.roles_id]
        );
        if (!roleRows.length) throw new Error('Role not found');
        const role = roleRows[0];

        // Step 3: Determine User IDs
        let filterUserIds = [];
        if (userIds && Array.isArray(userIds) && userIds.length > 0) {
            filterUserIds = userIds;
        } else {
            // fallback to current user only
            filterUserIds = [userId];
        }

        // Step 4: Build Dynamic SQL with placeholders for each item in arrays
        let baseQuery = `
            SELECT  
                c.*,  
                b.name,  
                u.name AS username,  
                u.emp_id,  
                com.company_name,  
                com.company_id,  
                tic.project_id AS workid,  
                br.branch_name,  
                con.contact_name,  
                tic.project_name,  
                t.activity_name,  
                tca.package_name
            FROM tasks c
            LEFT JOIN users b ON c.approved_by = b.id
            LEFT JOIN users u ON c.user_id = u.id
            LEFT JOIN companies com ON c.company_id = com.id
            LEFT JOIN branches br ON com.branch_id = br.id
            LEFT JOIN contacts con ON c.contact_id = con.id
            LEFT JOIN projects tic ON c.project_id = tic.id
            LEFT JOIN packages tca ON tic.package = tca.id
            LEFT JOIN tasktypes t ON c.tasktype_id = t.id
            WHERE c.tasktype_id NOT IN (1,8)
        `;

        const queryParams = [];

        // User filter
        if (filterUserIds && filterUserIds.length > 0) {
            baseQuery += ` AND c.user_id IN (${filterUserIds.map(() => '?').join(',')})`;
            queryParams.push(...filterUserIds);
        }

        // Branch filter
        if (branch && Array.isArray(branch) && branch.length > 0) {
            baseQuery += ` AND u.branch_id IN (${branch.map(() => '?').join(',')})`;
            queryParams.push(...branch);
        } else if (branch && !Array.isArray(branch)) {
            baseQuery += ' AND u.branch_id = ?';
            queryParams.push(branch);
        }

        // Date range filter
        if (startDate && endDate) {
            const start = startDate.length === 10 ? `${startDate} 00:00:00` : startDate;
            const end = endDate.length === 10 ? `${endDate} 23:59:59` : endDate;
            baseQuery += ' AND c.checkin_at BETWEEN ? AND ?';
            queryParams.push(start, end);
        }

        // Add ordering
        baseQuery += ' ORDER BY c.checkin_at DESC';

        // Step 5: Execute Query
        const rows = await db.query(baseQuery, queryParams);

        // Check if rows is undefined or not an array
        if (!rows || !Array.isArray(rows)) {
            return [];
        }

        // Step 6: Format Result
        const result = await Promise.all((rows || []).map(async row => {
            let ack = [];
            let approved = [];
            try {
                if (row.ack_contact_id) {
                    ack = await db.query('SELECT contact_name FROM contacts WHERE id = ?', [row.ack_contact_id]);
                }
                if (row.approved_by) {
                    approved = await db.query('SELECT name FROM users WHERE id = ?', [row.approved_by]);
                }
            } catch (error) {
                // ignore
            }
            return {
                taskid: row.id,
                filldate: row.checkin_at ? new Date(row.checkin_at).toLocaleString('en-GB') : '--',
                employeeid: row.emp_id,
                empname: row.username,
                checkout_location: row.checkin_location,
                checkoutat: row.checkout_at || '--',
                status: row.status,
                approvedby: approved && approved[0] ? approved[0].name : '--',
                approvedtime: row.approved_date ? new Date(row.approved_date).toLocaleString('en-GB') : '--',
                remarks: row.remarks || '--',
                companyname: row.company_name,
                customerid: row.company_id,
                startdate: row.start_date ? new Date(row.start_date).toLocaleDateString('en-GB') : '--',
                tasktype: row.activity_name,
                ackby: ack && ack[0] ? ack[0].contact_name : '--',
                branch: row.branch_name,
                workid: row.workid,
                project_name: row.project_name,
                package_name: row.package_name
            };
        }));

        return result;
    } catch (error) {
        throw error;
    }
}

export class ActionReport {
    static async datatableList({ page = 1, limit = 10, sortBy = 'checkin_at', sortOrder = 'DESC', search = '', userIds = [], branch = [], startDate, endDate }) {
        const allowedSortFields = ['checkin_at', 'emp_id', 'username', 'company_name', 'branch_name', 'status', 'activity_name', 'project_name'];
        const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'checkin_at';
        const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
        const pageNum = parseInt(page);
        const limitNum = parseInt(limit);
        const offset = (pageNum - 1) * limitNum;

        let baseQuery = `SELECT  c.*,  b.name,  u.name AS username,  u.emp_id,  com.company_name,  com.company_id AS company_id_ref,  tic.project_id AS workid,  br.branch_name,  con.contact_name,  tic.project_name,  t.activity_name,  tca.package_name FROM tasks c LEFT JOIN users b ON c.approved_by = b.id LEFT JOIN users u ON c.user_id = u.id LEFT JOIN companies com ON c.company_id = com.id LEFT JOIN branches br ON com.branch_id = br.id LEFT JOIN contacts con ON c.contact_id = con.id LEFT JOIN projects tic ON c.project_id = tic.id LEFT JOIN packages tca ON tic.package = tca.id LEFT JOIN tasktypes t ON c.tasktype_id = t.id WHERE c.tasktype_id NOT IN (1,8)`;
        let params = [];

        // User filter
        if (userIds && Array.isArray(userIds) && userIds.length > 0) {
            baseQuery += ` AND c.user_id IN (${userIds.map(() => '?').join(',')})`;
            params.push(...userIds);
        }

        // Branch filter
        if (branch && Array.isArray(branch) && branch.length > 0) {
            baseQuery += ` AND u.branch_id IN (${branch.map(() => '?').join(',')})`;
            params.push(...branch);
        } else if (branch && !Array.isArray(branch)) {
            baseQuery += ' AND u.branch_id = ?';
            params.push(branch);
        }

        // Date range filter
        if (startDate && endDate) {
            const start = startDate.length === 10 ? `${startDate} 00:00:00` : startDate;
            const end = endDate.length === 10 ? `${endDate} 23:59:59` : endDate;
            baseQuery += ' AND c.checkin_at BETWEEN ? AND ?';
            params.push(start, end);
        }

        if (search && search.trim()) {
            const likeSearch = `%${search.trim()}%`;
            baseQuery += ` AND (u.emp_id LIKE ? OR u.name LIKE ? OR com.company_name LIKE ? OR br.branch_name LIKE ? OR c.status LIKE ? OR t.activity_name LIKE ? OR tic.project_name LIKE ?)`;
            params.push(likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch);
        }

        // Get total count
        let countBaseQuery = baseQuery;
        let countQuery = `SELECT COUNT(*) as total FROM (${countBaseQuery} GROUP BY c.id) as sub`;
        const countResult = await db.query(countQuery, params);
        const total = countResult[0]?.total || 0;

        // Add ORDER BY, LIMIT, OFFSET
        baseQuery += ` GROUP BY c.id ORDER BY c.${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
        params.push(limitNum, offset);

        const reports = await db.query(baseQuery, params);
        return { reports, total };
    }
}



