import { db } from "../utils/database.js";

async function findAllAttendance(startDate = null, endDate = null) {
    try {
        let query = `
            SELECT t.*, u.emp_id, u.name, b.branch_name
            FROM tasks t
            LEFT JOIN users u ON t.user_id = u.id
            LEFT JOIN branches b ON u.branch_id = b.id
            WHERE t.tasktype_id = 1
        `;
        let params = [];

        // Add date filtering if provided
        if (startDate && endDate) {
            query += " AND DATE(t.start_date) BETWEEN ? AND ?";
            params.push(startDate, endDate);
        } else if (startDate) {
            query += " AND DATE(t.start_date) >= ?";
            params.push(startDate);
        } else if (endDate) {
            query += " AND DATE(t.start_date) <= ?";
            params.push(endDate);
        }

        const attendance = await db.query(query, params);
        return attendance;
    } catch (error) {
        console.error("Error fetching attendance:", error);
        throw error;
    }
}

async function findAttendanceById(id) {
    try {
        const attendance = await db.query(
            `SELECT t.*, u.emp_id, u.name, b.branch_name
             FROM tasks t
             LEFT JOIN users u ON t.user_id = u.id
             LEFT JOIN branches b ON u.branch_id = b.id
             WHERE t.tasktype_id = 1 AND t.id = ?`,
            [id]
        );
        return attendance;  
    } catch (error) {
        console.error("Error fetching attendance by id:", error);
        throw error;
    }
}
async function viewlistAttendancePresentAbsent({ startDate, endDate, scope, userId, branchId }) {
    try {
        let query = `
            SELECT t.*, u.emp_id, u.name, b.branch_name,
                   CASE
                       WHEN t.status IN ('Closed', 'Inprogress') THEN 'present'
                       WHEN t.status = 'Open' THEN 'absent'
                       ELSE 'unknown'
                   END as attendance_status
            FROM tasks t
            LEFT JOIN users u ON t.user_id = u.id
            LEFT JOIN branches b ON u.branch_id = b.id
            WHERE t.tasktype_id = 1
        `;
        let params = [];

        // Role-based filtering (same logic as attendancerolewisemodel.js)
        if (scope === 'owned') {
            query += " AND t.user_id = ?";
            params.push(userId);
        } else if (scope === 'branch') {
            query += " AND u.branch_id = ?";
            params.push(branchId);
        }
        // If scope === 'global', no extra filter (all records)

        // Add date filtering
        if (startDate && endDate) {
            query += " AND DATE(t.start_date) BETWEEN ? AND ?";
            params.push(startDate, endDate);
        } else if (startDate) {
            query += " AND DATE(t.start_date) >= ?";
            params.push(startDate);
        } else if (endDate) {
            query += " AND DATE(t.start_date) <= ?";
            params.push(endDate);
        }

        query += " ORDER BY t.start_date DESC, u.name ASC";

        let attendanceData = await db.query(query, params);

        // Separate present and absent lists
        const presentList = attendanceData.filter(record =>
            record.status === 'Closed' || record.status === 'Inprogress'
        );

        const absentList = attendanceData.filter(record =>
            record.status === 'Open'
        );

        return {
            present: presentList,
            absent: absentList,
            total: attendanceData.length,
            presentCount: presentList.length,
            absentCount: absentList.length
        };
    } catch (error) {
        console.error("Error fetching attendance present/absent list:", error);
        throw error;
    }
}
export {
    findAllAttendance,
    findAttendanceById,
    viewlistAttendancePresentAbsent
}