import { db } from "../utils/database.js";

function toNull(val) {
    return val === undefined ? null : val;
}

export async function findAttendanceRolewise({ scope, userId, branchId, startDate, endDate }) {
    try {
        let query = `
            SELECT t.*, u.emp_id, u.name, b.branch_name
            FROM tasks t
            LEFT JOIN users u ON t.user_id = u.id
            LEFT JOIN branches b ON u.branch_id = b.id
            WHERE t.tasktype_id = 1
        `;
        let params = [];

        // Role-based filtering
        if (scope === 'owned') {
            query += " AND t.user_id = ?";
            params.push(toNull(userId));
        } else if (scope === 'branch') {
            query += " AND u.branch_id = ?";
            params.push(toNull(branchId));
        }
        // If global, no extra filter

        // Date filter
        if (startDate) {
            query += " AND DATE(t.start_date) >= ?";
            params.push(toNull(startDate));
        }
        if (endDate) {
            query += " AND DATE(t.start_date) <= ?";
            params.push(toNull(endDate));
        }

        query += " ORDER BY t.start_date DESC";

        let attendance = await db.query(query, params);
        return attendance;
    } catch (error) {
        console.error("Error fetching attendance rolewise:", error);
        throw error;
    }
}

export class AttendanceRolewiseModel {
    /*static async datatableList({ page = 1, limit = 10, sortBy = 'start_date', sortOrder = 'DESC', search = '', startDate, endDate }) {
        const allowedSortFields = ['start_date', 'emp_id', 'name', 'branch_name', 'status'];
        const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'start_date';
        const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
        const pageNum = parseInt(page);
        const limitNum = parseInt(limit);
        const offset = (pageNum - 1) * limitNum;

        let baseQuery = `SELECT t.*, u.emp_id, u.name, b.branch_name FROM tasks t LEFT JOIN users u ON t.user_id = u.id LEFT JOIN branches b ON u.branch_id = b.id WHERE t.tasktype_id = 1`;
        let params = [];

        if (search && search.trim()) {
            const likeSearch = `%${search.trim()}%`;
            baseQuery += ` AND (u.emp_id LIKE ? OR u.name LIKE ? OR b.branch_name LIKE ? OR t.status LIKE ?)`;
            params.push(likeSearch, likeSearch, likeSearch, likeSearch);
        }
        if (startDate) {
            baseQuery += ' AND DATE(t.start_date) >= ?';
            params.push(startDate);
        }
        if (endDate) {
            baseQuery += ' AND DATE(t.start_date) <= ?';
            params.push(endDate);
        }
        // Get total count
        const countQuery = baseQuery.replace(/^SELECT t\.*?, u\.emp_id, u\.name, b\.branch_name/, 'SELECT COUNT(*) as total');
        const countResult = await db.query(countQuery, params);
        const total = countResult[0]?.total || 0;

        // Add ORDER BY, LIMIT, OFFSET
        baseQuery += ` ORDER BY t.${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
        params.push(limitNum, offset);

        const attendance = await db.query(baseQuery, params);
        return { attendance, total };
    }*/
    static async datatableList({
    page = 1,
    limit = 10,
    sortBy = 'start_date',
    sortOrder = 'DESC',
    search = '',
    startDate,
    endDate,
    scope,
    userId,
    branchId
}) {
    const allowedSortFields = ['start_date', 'emp_id', 'name', 'branch_name', 'status'];
    const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'start_date';
    const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
    const pageNum = parseInt(page);
    const limitNum = parseInt(limit);
    const offset = (pageNum - 1) * limitNum;

    let baseQuery = `
        SELECT t.*, u.emp_id, u.name, b.branch_name
        FROM tasks t
        LEFT JOIN users u ON t.user_id = u.id
        LEFT JOIN branches b ON u.branch_id = b.id
        WHERE t.tasktype_id = 1
    `;
    let params = [];

    // 🔍 Role-based filtering
    if (scope === 'owned') {
        baseQuery += " AND t.user_id = ?";
        params.push(toNull(userId));
    } else if (scope === 'branch') {
        baseQuery += " AND u.branch_id = ?";
        params.push(toNull(branchId));
    }

    // 🔍 Search filter
    if (search && search.trim()) {
        const likeSearch = `%${search.trim()}%`;
        baseQuery += ` AND (u.emp_id LIKE ? OR u.name LIKE ? OR b.branch_name LIKE ? OR t.status LIKE ?)`;
        params.push(likeSearch, likeSearch, likeSearch, likeSearch);
    }

    // 📅 Date filter
    if (startDate) {
        baseQuery += ' AND DATE(t.start_date) >= ?';
        params.push(startDate);
    }
    if (endDate) {
        baseQuery += ' AND DATE(t.start_date) <= ?';
        params.push(endDate);
    }

    // 🧮 Total count
    const countQuery = `SELECT COUNT(*) as total FROM (${baseQuery}) as sub`;
    const countResult = await db.query(countQuery, params);
    const total = countResult[0]?.total || 0;

    // ✅ Present count (status != 'Open')
    const presentQuery = `SELECT COUNT(*) as present FROM (${baseQuery} AND t.status != 'Open') as sub`;
    const presentResult = await db.query(presentQuery, params);
    const present = presentResult[0]?.present || 0;

    // ❌ Absent count (status = 'Open')
    const absentQuery = `SELECT COUNT(*) as absent FROM (${baseQuery} AND t.status = 'Open') as sub`;
    const absentResult = await db.query(absentQuery, params);
    const absent = absentResult[0]?.absent || 0;

    // 📦 Pagination
    baseQuery += ` ORDER BY t.${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
    params.push(limitNum, offset);

    const attendance = await db.query(baseQuery, params);
    return { attendance, total, present, absent };
}
}