import { db } from '../utils/database.js';

export async function findAllCheckoutReports({ branch, userIds, startdate, enddate }) {
  let baseQuery = `
    SELECT
      c.*, b.name, u.name AS username, u.emp_id AS employee_id,
      com.company_name, com.company_address, com.company_id,
      tic.project_id AS ticket_id, br.branch_name, con.contact_name,
      tic.project_name AS title, t.activity_name, tca.package_name AS category
    FROM tasks c
    LEFT JOIN users b ON c.approved_by = b.id
    LEFT JOIN users u ON c.user_id = u.id
    LEFT JOIN companies com ON c.company_id = com.id
    LEFT JOIN branches br ON com.branch_id = br.id
    LEFT JOIN contacts con ON c.contact_id = con.id
    LEFT JOIN projects tic ON c.project_id = tic.id
    LEFT JOIN packages tca ON tic.package = tca.id
    LEFT JOIN tasktypes t ON c.tasktype_id = t.id
    WHERE c.tasktype_id = 3
      AND c.checkin_at BETWEEN ? AND ?
  `;

  const params = [
    `${startdate} 00:00:00`,
    `${enddate} 23:59:59`
  ];

  // Add user filter only if userIds array has values
  if (userIds && userIds.length > 0) {
    baseQuery += ` AND c.user_id IN (${userIds.map(() => '?').join(',')})`;
    params.push(...userIds);
  }

  // Add branch filter only if branch array has values
  if (branch && branch.length > 0) {
    baseQuery += ` AND u.branch_id IN (${branch.map(() => '?').join(',')})`;
    params.push(...branch);
  }

  baseQuery += ` ORDER BY c.checkin_at DESC`;

  const rows = await db.query(baseQuery, params);

  // Format response
  const data = (rows || []).map((deviation, i) => {
    // Time calculations
    let timespend = 0, manualtimespend = 0, actualtimespend = 0;
    if (deviation.checkout_at && deviation.checkin_at) {
      const time = new Date(deviation.checkout_at) - new Date(deviation.checkin_at);
      timespend = time > 0 ? new Date(time).toISOString().substr(11, 8) : '00:00:00';
    }
    if (deviation.manual_checkout_at && deviation.checkout_at) {
      const time = new Date(deviation.checkout_at) - new Date(deviation.manual_checkout_at);
      manualtimespend = time > 0 ? new Date(time).toISOString().substr(11, 8) : '00:00:00';
    }
    if (deviation.manual_checkout_at && deviation.checkin_at) {
      const actualtime = new Date(deviation.manual_checkout_at) - new Date(deviation.checkin_at);
      actualtimespend = actualtime > 0 ? new Date(actualtime).toISOString().substr(11, 8) : '00:00:00';
    } else if (deviation.checkout_at && deviation.checkin_at) {
      const actualtime = new Date(deviation.checkout_at) - new Date(deviation.checkin_at);
      actualtimespend = actualtime > 0 ? new Date(actualtime).toISOString().substr(11, 8) : '00:00:00';
    }

    return {
      sno: i + 1,
      taskid: deviation.id,
      tickettemp: 'Ticket V2',
      projectid: deviation.ticket_id,
      filldate: deviation.start_date ? new Date(deviation.start_date).toLocaleDateString('en-GB') : '--',
      employeeid: deviation.employee_id,
      empname: deviation.username,
      company_address: deviation.company_address,
      checkin_location: deviation.checkin_location,
      checkindate: deviation.checkin_at ? new Date(deviation.checkin_at).toLocaleDateString('en-GB') : '--',
      checkintime: deviation.checkin_at ? new Date(deviation.checkin_at).toLocaleTimeString('en-GB') : '--',
      checkoutdate: deviation.checkout_at ? new Date(deviation.checkout_at).toLocaleDateString('en-GB') : '--',
      checkouttime: deviation.checkout_at ? new Date(deviation.checkout_at).toLocaleTimeString('en-GB') : '--',
      checkout_location: deviation.checkout_location,
      checkoutat: deviation.checkout_at || '--',
      force: deviation.force ? 'Yes' : 'No',
      forcelat: deviation.force ? deviation.checkout_lat : '--',
      forcelong: deviation.force ? deviation.checkout_long : '--',
      status: deviation.status,
      deviation_in: deviation.deviation_in,
      deviation_out: deviation.deviation_out,
      task: deviation.task,
      timespend,
      actualtimespend,
      manualtimespend,
      FormID: '',
      FormIdentifiers: '',
      approvedby: '',
      approvedtime: '',
      remarks: '',
      companyname: deviation.company_name,
      customerid: deviation.company_id,
      startdate: deviation.start_date ? new Date(deviation.start_date).toLocaleDateString('en-GB') : '--',
      tasktype: deviation.activity_name,
      packagename: deviation.category,
      projecttitle: deviation.title,
      workid: ''
    };
  });

  return data;
}

export class CheckoutReport {
    static async datatableList({ page = 1, limit = 10, sortBy = 'checkin_at', sortOrder = 'DESC', search = '', userIds = [], branch = [], startdate, enddate }) {
        const allowedSortFields = ['checkin_at', 'employee_id', 'username', 'company_name', 'branch_name', 'status', 'activity_name', 'title'];
        const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'checkin_at';
        const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
        const pageNum = parseInt(page);
        const limitNum = parseInt(limit);
        const offset = (pageNum - 1) * limitNum;

        let baseQuery = `SELECT c.*, b.name, u.name AS username, u.emp_id AS employee_id, com.company_name, com.company_address, com.company_id, tic.project_id AS ticket_id, br.branch_name, con.contact_name, tic.project_name AS title, t.activity_name, tca.package_name AS category FROM tasks c LEFT JOIN users b ON c.approved_by = b.id LEFT JOIN users u ON c.user_id = u.id LEFT JOIN companies com ON c.company_id = com.id LEFT JOIN branches br ON com.branch_id = br.id LEFT JOIN contacts con ON c.contact_id = con.id LEFT JOIN projects tic ON c.project_id = tic.id LEFT JOIN packages tca ON tic.package = tca.id LEFT JOIN tasktypes t ON c.tasktype_id = t.id WHERE c.tasktype_id = 3`;
        let params = [];

        // Date range filter
        if (startdate && enddate) {
            baseQuery += ' AND c.checkin_at BETWEEN ? AND ?';
            params.push(`${startdate} 00:00:00`, `${enddate} 23:59:59`);
        }

        // User filter
        if (userIds && Array.isArray(userIds) && userIds.length > 0) {
            baseQuery += ` AND c.user_id IN (${userIds.map(() => '?').join(',')})`;
            params.push(...userIds);
        }

        // Branch filter
        if (branch && Array.isArray(branch) && branch.length > 0) {
            baseQuery += ` AND u.branch_id IN (${branch.map(() => '?').join(',')})`;
            params.push(...branch);
        }

        if (search && search.trim()) {
            const likeSearch = `%${search.trim()}%`;
            baseQuery += ` AND (u.emp_id LIKE ? OR u.name LIKE ? OR com.company_name LIKE ? OR br.branch_name LIKE ? OR c.status LIKE ? OR t.activity_name LIKE ? OR tic.project_name LIKE ?)`;
            params.push(likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch);
        }

        // Get total count
        let countBaseQuery = baseQuery;
        let countQuery = `SELECT COUNT(*) as total FROM (${countBaseQuery} GROUP BY c.id) as sub`;
        const countResult = await db.query(countQuery, params);
        const total = countResult[0]?.total || 0;

        // Add ORDER BY, LIMIT, OFFSET
        baseQuery += ` GROUP BY c.id ORDER BY c.${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
        params.push(limitNum, offset);

        const rows = await db.query(baseQuery, params);
        // Format response
        const data = (rows || []).map((deviation, i) => {
            // Time calculations
            let timespend = 0, manualtimespend = 0, actualtimespend = 0;
            if (deviation.checkout_at && deviation.checkin_at) {
                const time = new Date(deviation.checkout_at) - new Date(deviation.checkin_at);
                timespend = time > 0 ? new Date(time).toISOString().substr(11, 8) : '00:00:00';
            }
            if (deviation.manual_checkout_at && deviation.checkout_at) {
                const time = new Date(deviation.checkout_at) - new Date(deviation.manual_checkout_at);
                manualtimespend = time > 0 ? new Date(time).toISOString().substr(11, 8) : '00:00:00';
            }
            if (deviation.manual_checkout_at && deviation.checkin_at) {
                const actualtime = new Date(deviation.manual_checkout_at) - new Date(deviation.checkin_at);
                actualtimespend = actualtime > 0 ? new Date(actualtime).toISOString().substr(11, 8) : '00:00:00';
            } else if (deviation.checkout_at && deviation.checkin_at) {
                const actualtime = new Date(deviation.checkout_at) - new Date(deviation.checkin_at);
                actualtimespend = actualtime > 0 ? new Date(actualtime).toISOString().substr(11, 8) : '00:00:00';
            }

            return {
                sno: i + 1 + offset,
                taskid: deviation.id,
                tickettemp: 'Ticket V2',
                projectid: deviation.ticket_id,
                filldate: deviation.start_date ? new Date(deviation.start_date).toLocaleDateString('en-GB') : '--',
                employeeid: deviation.employee_id,
                empname: deviation.username,
                company_address: deviation.company_address,
                checkin_location: deviation.checkin_location,
                checkindate: deviation.checkin_at ? new Date(deviation.checkin_at).toLocaleDateString('en-GB') : '--',
                checkintime: deviation.checkin_at ? new Date(deviation.checkin_at).toLocaleTimeString('en-GB') : '--',
                checkoutdate: deviation.checkout_at ? new Date(deviation.checkout_at).toLocaleDateString('en-GB') : '--',
                checkouttime: deviation.checkout_at ? new Date(deviation.checkout_at).toLocaleTimeString('en-GB') : '--',
                checkout_location: deviation.checkout_location,
                checkoutat: deviation.checkout_at || '--',
                force: deviation.force ? 'Yes' : 'No',
                forcelat: deviation.force ? deviation.checkout_lat : '--',
                forcelong: deviation.force ? deviation.checkout_long : '--',
                status: deviation.status,
                deviation_in: deviation.deviation_in,
                deviation_out: deviation.deviation_out,
                task: deviation.task,
                timespend,
                actualtimespend,
                manualtimespend,
                FormID: '',
                FormIdentifiers: '',
                approvedby: '',
                approvedtime: '',
                remarks: '',
                companyname: deviation.company_name,
                customerid: deviation.company_id,
                startdate: deviation.start_date ? new Date(deviation.start_date).toLocaleDateString('en-GB') : '--',
                tasktype: deviation.activity_name,
                packagename: deviation.category,
                projecttitle: deviation.title,
                workid: ''
            };
        });
        return { data, total };
    }
} 
