import { db } from '../utils/database.js';

export async function findAllClaimReports({ branch, userIds, startdate, enddate }) {
  // Build base query
  let baseQuery = `
    SELECT
      c.*,
      b.name,
      u.name AS username,
      u.emp_id AS employee_id,
      com.company_name,
      com.company_id AS compid,
      tic.project_id AS workid,
      br.branch_name,
      con.contact_name,
      tic.project_name AS title,
      t.activity_name,
      tca.package_name AS category
    FROM tasks c
    LEFT JOIN users b ON c.approved_by = b.id
    LEFT JOIN users u ON c.user_id = u.id
    LEFT JOIN companies com ON c.company_id = com.id
    LEFT JOIN branches br ON com.branch_id = br.id
    LEFT JOIN contacts con ON c.contact_id = con.id
    LEFT JOIN projects tic ON c.project_id = tic.id
    LEFT JOIN packages tca ON tic.package = tca.id
    LEFT JOIN tasktypes t ON c.tasktype_id = t.id
    WHERE c.tasktype_id IN (1,3,8)
      AND c.checkin_at BETWEEN ? AND ?
  `;

  const params = [`${startdate} 00:00:00`, `${enddate} 23:59:59`];

  // Add user filter only if userIds array has values
  if (userIds && userIds.length > 0) {
    baseQuery += ` AND c.user_id IN (${userIds.map(() => '?').join(',')})`;
    params.push(...userIds);
  }

  // Add branch filter only if branch array has values
  if (branch && branch.length > 0) {
    baseQuery += ` AND u.branch_id IN (${branch.map(() => '?').join(',')})`;
    params.push(...branch);
  }

  baseQuery += ` ORDER BY c.checkin_at DESC`;

  const rows = await db.query(baseQuery, params);

  // Format response
  const data = await Promise.all((rows || []).map(async (deviation, i) => {
    // Find previous record for checkinlocation
    let previousRecord = null;
    try {
      const prevRows = await db.query(
        `SELECT checkin_location FROM tasks
         WHERE user_id = ? AND start_date = ? AND tasktype_id IN (1,3,8)
         AND checkin_at < (SELECT checkin_at FROM tasks WHERE id = ?)
         ORDER BY checkin_at DESC LIMIT 1`,
        [deviation.user_id, deviation.start_date, deviation.id]
      );
      previousRecord = prevRows[0];
    } catch {}

    return {
      sno: i + 1,
      status: deviation.claim_status,
      approvedby: deviation.name,
      approveddate: deviation.approved_date
        ? new Date(deviation.approved_date).toLocaleDateString('en-GB')
        : '--',
      remarks: deviation.claim_approved_remarks,
      companyid: deviation.compid,
      companyname: deviation.company_name,
      activityname: deviation.activity_name,
      taskdate: deviation.checkin_at
        ? new Date(deviation.checkin_at).toLocaleString('en-GB', { hour12: false })
        : '--',
      checkinlocation: previousRecord
        ? previousRecord.checkin_location
        : deviation.checkin_location,
      checkoutlocation: deviation.checkout_location,
      checkout_at: deviation.claim_created_date
        ? new Date(deviation.claim_created_date).toLocaleString('en-GB', { hour12: false })
        : deviation.checkin_at,
      ticket_id: deviation.workid,
      distance_travelled: deviation.distance_travelled,
      transport: deviation.transport_mode,
      claim_amount: deviation.claim_amount,
      branch_name: deviation.branch_name,
      claim_remarks: deviation.claim_remarks,
      category: deviation.category,
      employee_id: deviation.employee_id,
      username: deviation.username,
      action: `<a style='cursor:pointer' class='text-success viewstatus' id='${deviation.id}'><i class='fa fa-eye'></i></a>`
    };
  }));

  return data;
}

export class ClaimReport {
    static async datatableList({ page = 1, limit = 10, sortBy = 'checkin_at', sortOrder = 'DESC', search = '', userIds = [], branch = [], startdate, enddate }) {
        const allowedSortFields = ['checkin_at', 'employee_id', 'username', 'company_name', 'branch_name', 'status', 'activity_name', 'title', 'claim_amount'];
        const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'checkin_at';
        const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
        const pageNum = parseInt(page);
        const limitNum = parseInt(limit);
        const offset = (pageNum - 1) * limitNum;

        let baseQuery = `SELECT c.*, b.name, u.name AS username, u.emp_id AS employee_id, com.company_name, com.company_id AS compid, tic.project_id AS workid, br.branch_name, con.contact_name, tic.project_name AS title, t.activity_name, tca.package_name AS category FROM tasks c LEFT JOIN users b ON c.approved_by = b.id LEFT JOIN users u ON c.user_id = u.id LEFT JOIN companies com ON c.company_id = com.id LEFT JOIN branches br ON com.branch_id = br.id LEFT JOIN contacts con ON c.contact_id = con.id LEFT JOIN projects tic ON c.project_id = tic.id LEFT JOIN packages tca ON tic.package = tca.id LEFT JOIN tasktypes t ON c.tasktype_id = t.id WHERE c.tasktype_id IN (1,3,8)`;
        let params = [];

        // Date range filter
        if (startdate && enddate) {
            baseQuery += ' AND c.checkin_at BETWEEN ? AND ?';
            params.push(`${startdate} 00:00:00`, `${enddate} 23:59:59`);
        }

        // User filter
        if (userIds && Array.isArray(userIds) && userIds.length > 0) {
            baseQuery += ` AND c.user_id IN (${userIds.map(() => '?').join(',')})`;
            params.push(...userIds);
        }

        // Branch filter
        if (branch && Array.isArray(branch) && branch.length > 0) {
            baseQuery += ` AND u.branch_id IN (${branch.map(() => '?').join(',')})`;
            params.push(...branch);
        }

        if (search && search.trim()) {
            const likeSearch = `%${search.trim()}%`;
            baseQuery += ` AND (u.emp_id LIKE ? OR u.name LIKE ? OR com.company_name LIKE ? OR br.branch_name LIKE ? OR c.status LIKE ? OR t.activity_name LIKE ? OR tic.project_name LIKE ? OR c.claim_amount LIKE ?)`;
            params.push(likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch);
        }

        // Get total count
        let countBaseQuery = baseQuery;
        let countQuery = `SELECT COUNT(*) as total FROM (${countBaseQuery} GROUP BY c.id) as sub`;
        const countResult = await db.query(countQuery, params);
        const total = countResult[0]?.total || 0;

        // Add ORDER BY, LIMIT, OFFSET
        baseQuery += ` GROUP BY c.id ORDER BY c.${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
        params.push(limitNum, offset);

        const reports = await db.query(baseQuery, params);
        return { reports, total };
    }
} 
