import { db } from '../utils/database.js';

export class Contact {
    static async create(contactData, userId) {
        const query = `
            INSERT INTO contacts (
                contact_name, email, mobile_no, create_date,
                designation, company_id, status, created_at, updated_at
            ) VALUES (?, ?, ?, NOW(), ?, ?, 1, NOW(), NOW())
        `;
        
        const result = await db.query(query, [
            contactData.contact_name,
            contactData.email,
            contactData.mobile_no,
            contactData.designation,
            contactData.company_id
        ]);
        return result.insertId;
    }
    static async update(id, contactData) {
        const query = `
            UPDATE contacts 
            SET contact_name = ?,
                email = ?,
                mobile_no = ?,
                designation = ?,
                company_id = ?,
                updated_at = NOW()
            WHERE id = ? AND status = 1
        `;
        
        return db.query(query, [
            contactData.contact_name,
            contactData.email,
            contactData.mobile_no,
            contactData.designation,
            contactData.company_id,
            id
        ]);
    }

    static async softDelete(id) {
        const query = `
            UPDATE contacts 
            SET status = 0,
                updated_at = NOW() 
            WHERE id = ?
        `;
        return db.query(query, [id]);
    }

    static async findById(id) {
        const query = `
            SELECT c.*, co.company_name 
            FROM contacts c
            LEFT JOIN companies co ON c.company_id = co.id
            WHERE c.id = ? AND c.status = 1
        `;
        const [contact] = await db.query(query, [id]);
        return contact;
    }

    static async list() {
        const query = `
            SELECT c.*, co.company_name 
            FROM contacts c
            LEFT JOIN companies co ON c.company_id = co.id
            WHERE c.status = 1 
            ORDER BY c.created_at DESC
        `;
        return db.query(query);
    }
    
    static async datatableList({ page = 1, limit = 10, sortBy = 'created_at', sortOrder = 'DESC', search = '' }) {
        const allowedSortFields = ['created_at', 'name', 'email', 'mobile'];
        const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'created_at';
        const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
        const pageNum = parseInt(page);
        const limitNum = parseInt(limit);
        const offset = (pageNum - 1) * limitNum;

        let baseQuery = `SELECT * FROM contacts`;
        let params = [];

        if (search && search.trim()) {
            const likeSearch = `%${search.trim()}%`;
            baseQuery += ` WHERE (contact_name LIKE ? OR email LIKE ? OR mobile_no LIKE ? OR designation LIKE ?)`;
            params.push(likeSearch, likeSearch, likeSearch, likeSearch);
        }

        // Get total count
        const countQuery = baseQuery.replace(/^SELECT \* FROM contacts/, 'SELECT COUNT(*) as total FROM contacts');
        const countResult = await db.query(countQuery, params);
        const total = countResult[0]?.total || 0;

        // Add ORDER BY, LIMIT, OFFSET
        baseQuery += ` ORDER BY ${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
        params.push(limitNum, offset);

        const contacts = await db.query(baseQuery, params);
        return { contacts, total };
    }
    static async checkContactExists(email, mobile, excludeId = null) {
        try {
            const query = `
                SELECT COUNT(*) as count 
                FROM contacts 
                WHERE (email = ? OR mobile_no = ?)
                AND status = 1
                ${excludeId ? 'AND id != ?' : ''}
            `;
            const params = excludeId ? [email, mobile, excludeId] : [email, mobile];
            const [[result]] = await db.query(query, params);
            return result.count > 0;
        } catch (error) {
            console.error('Check contact exists error:', error);
            return false;
        }
    }

    static async listActiveByCompanyId(companyId) {
        const query = `
            SELECT c.*, co.company_name
            FROM contacts c
            LEFT JOIN companies co ON c.company_id = co.id
            WHERE c.company_id = ? AND c.status = 1
            ORDER BY c.created_at DESC
        `;
        return db.query(query, [companyId]);
    }
   
}