import { db } from '../utils/database.js';

export const getContactById = async (contactId) => {
    let rows = await db.query(
        'SELECT * FROM contacts WHERE id = ?',
        [contactId]
    );
    console.log('DB rows:', rows); // Add this line
    console.log('Returning:', rows[0]);
    return rows || null;
};

export const getCustomersByContact = async (contactId) => {
    // Get the company_id from the contact
    let contactRows = await db.query(
        'SELECT company_id FROM contacts WHERE id = ?',
        [contactId]
    );
    if (!contactRows.length) return [];
    const companyId = contactRows[0].company_id;
    // Get the company details
    let companyRows = await db.query(
        'SELECT * FROM companies WHERE id = ?',
        [companyId]
    );
    return companyRows;
};

export const getProjectsByContact = async (contactId) => {
    let rows = await db.query(
        'SELECT * FROM projects WHERE contact_id = ?',
        [contactId]
    );

    // If no projects found, return empty array
    if (!rows || rows.length === 0) {
        return [];
    }

    // Get all project IDs for batch queries
    const projectIds = rows.map(project => project.id);
    const projectIdsPlaceholder = projectIds.map(() => '?').join(',');

    // Fetch all project assignees for these projects
    const assigneesQuery = `
        SELECT pa.project_id, pa.assigned_on, pa.assigned_by, pa.status,
               u.id as user_id, u.name as user_name, u.emp_id, u.email as user_email,
               u.mobile as user_mobile, u.emp_id as user_designation,
               r.name as user_role, r.scope as user_role_scope,
               ab.name as assigned_by_name, ab.emp_id as assigned_by_emp_id,
               ab.email as assigned_by_email, ab.mobile as assigned_by_mobile,
               abr.name as assigned_by_role, abr.scope as assigned_by_role_scope
        FROM projectassignees pa
        LEFT JOIN users u ON pa.user_id = u.id
        LEFT JOIN roles r ON u.roles_id = r.id
        LEFT JOIN users ab ON pa.assigned_by = ab.id
        LEFT JOIN roles abr ON ab.roles_id = abr.id
        WHERE pa.project_id IN (${projectIdsPlaceholder})
        ORDER BY pa.assigned_on DESC
    `;
    const allAssignees = await db.query(assigneesQuery, projectIds);

    // Group assignees by project_id
    const assigneesByProjectId = {};
    allAssignees.forEach(assignee => {
        if (!assigneesByProjectId[assignee.project_id]) {
            assigneesByProjectId[assignee.project_id] = [];
        }
        assigneesByProjectId[assignee.project_id].push({
            user_id: assignee.user_id,
            user_name: assignee.user_name,
            emp_id: assignee.emp_id,
            user_email: assignee.user_email,
            user_mobile: assignee.user_mobile,
            user_emp_id: assignee.user_designation,
            user_role: assignee.user_role,
            user_role_scope: assignee.user_role_scope,
            assigned_on: assignee.assigned_on,
            assigned_by: assignee.assigned_by,
            assigned_by_name: assignee.assigned_by_name,
            assigned_by_emp_id: assignee.assigned_by_emp_id,
            assigned_by_email: assignee.assigned_by_email,
            assigned_by_mobile: assignee.assigned_by_mobile,
            assigned_by_role: assignee.assigned_by_role,
            assigned_by_role_scope: assignee.assigned_by_role_scope,
            status: assignee.status
        });
    });

    // Merge assignees with projects
    const projectsWithAssignees = rows.map(project => ({
        ...project,
        project_assignees: assigneesByProjectId[project.id] || [],
        assignees_count: (assigneesByProjectId[project.id] || []).length
    }));

    return projectsWithAssignees;
};

export const getTasksByContact = async (contactId) => {
    let rows = await db.query(
        'SELECT * FROM tasks WHERE contact_id = ?',
        [contactId]
    );
    return rows;
};

export const getFilesByContact = async (contactId) => {
    // Placeholder: update with your actual files table/logic if needed
    return [];
};
