import { db } from '../utils/database.js';

export class Conveyance {
    static async getAdminConveyance({ start_date, end_date, branch_id }) {
        // Build user filter
        let userFilter = '';
        let params = [start_date, end_date];
        if (branch_id) {
            userFilter = 'AND u.branch_id = ?';
            params.push(branch_id);
        }

        // Get all users (filtered by branch if provided)
        let users = await db.query(
            `SELECT u.id, u.name, u.branch_id, b.branch_name
             FROM users u
             LEFT JOIN branches b ON u.branch_id = b.id
             WHERE u.active = 1 ${userFilter}`
            , branch_id ? [branch_id] : []
        );

        if (!users || users.length === 0) {
            return [];
        }

        // For each user, get conveyance details
        const conveyance = [];
        for (const user of users) {
            // Get all tasks for this user in date range
            let rows = await db.query(
                `SELECT 
                    DATE(t.start_date) as conveyance_date,
                    SUM(t.distance_travelled) as distance,
                    SUM(t.claim_amount) as amount,
                    SUM(t.time_spent) as timespent
                 FROM tasks t
                 WHERE t.user_id = ?
                   AND t.start_date BETWEEN ? AND ?
                   AND t.status = 'Closed'
                 GROUP BY DATE(t.start_date)`,
                [user.id, start_date, end_date]
            );

            // If user has no tasks, still add with zeroes
            if (!rows || rows.length === 0) {
                conveyance.push({
                    user_id: user.id,
                    user_name: user.name,
                    employee_id: user.employee_id,
                    branch_id: user.branch_id,
                    branch_name: user.branch_name,
                    conveyance: []
                });
            } else {
                conveyance.push({
                    user_id: user.id,
                    user_name: user.name,
                    employee_id: user.employee_id,
                    branch_id: user.branch_id,
                    branch_name: user.branch_name,
                    conveyance: rows.map(row => ({
                        conveyance_date: row.conveyance_date,
                        distance: Number(row.distance) || 0,
                        amount: Number(row.amount) || 0,
                        timespent: Number(row.timespent) || 0
                    }))
                });
            }
        }

        return conveyance;
    }

    // get all users conveyance
    static async findAllConveyance(startDate = null, endDate = null) {
    try {
        let query = `
            SELECT 
                u.id as emp_id,
                u.name,
                COUNT(CASE WHEN t.id IS NOT NULL THEN t.id END) as total_visits,
                COALESCE(SUM(CASE WHEN t.id IS NOT NULL THEN t.distance_travelled ELSE 0 END), 0) as total_distance,
                COALESCE(SUM(CASE WHEN t.id IS NOT NULL THEN t.claim_amount ELSE 0 END), 0) as total_claim_amount,
                COALESCE(SEC_TO_TIME(SUM(CASE WHEN t.id IS NOT NULL THEN t.time_spent ELSE 0 END) * 60), '00:00:00') as total_time_spent_minutes
            FROM users u
            LEFT JOIN tasks t ON u.id = t.user_id 
                AND t.tasktype_id != 1 
                AND t.status = 'Closed'
                AND t.checkout_at != ''
            LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
                AND tt.allow_checkin_checkout = '1'
        `;
        
        let params = [];
        let whereConditions = ["u.active = 1"];

        // Add date filtering if provided (using checkin_at instead of start_date)
        if (startDate && endDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) BETWEEN ? AND ?)");
            params.push(startDate, endDate);
        } else if (startDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) >= ?)");
            params.push(startDate);
        } else if (endDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) <= ?)");
            params.push(endDate);
        }

        // Add tasktype condition to ensure only tasks with allow_checkin_checkout = '1' are included
        whereConditions.push("(t.id IS NULL OR tt.allow_checkin_checkout = '1')");

        query += ` WHERE ${whereConditions.join(" AND ")}`;
        query += ` GROUP BY u.id, u.name ORDER BY u.name`;

        const conveyance = await db.query(query, params);
        return conveyance;
    } catch (error) {
        console.error("Error fetching conveyance:", error);
        throw error;
    }
}

    static async findConveyanceByUserId(userId, startDate = null, endDate = null) {
    try {
        let query = `
            SELECT 
                u.id as emp_id,
                u.name,
                t.id as task_id,
                t.task,
                t.start_date,
                t.start_time,
                t.end_time,
                t.distance_travelled as distance,
                t.claim_amount,
                t.time_spent as timespentmintus,
                SEC_TO_TIME(t.time_spent) as time_spent,
                t.status,
                t.checkin_at,
                t.checkin_location,
                t.checkout_at,
                t.checkout_location,
                t.checkin_lat,
                t.checkin_long
            FROM users u
            LEFT JOIN tasks t ON u.id = t.user_id 
                AND t.tasktype_id in (1, 3) 
                AND t.status = 'Closed'
                AND t.checkout_at != ''
            LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
                AND tt.allow_checkin_checkout = '1'
        `;
        
        let params = [userId];
        let whereConditions = ["u.id = ?", "u.active = 1"];

        // Add date filtering if provided (using checkin_at instead of start_date)
        if (startDate && endDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) BETWEEN ? AND ?)");
            params.push(startDate, endDate);
        } else if (startDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) >= ?)");
            params.push(startDate);
        } else if (endDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) <= ?)");
            params.push(endDate);
        }

        // Add tasktype condition to ensure only tasks with allow_checkin_checkout = '1' are included
        whereConditions.push("(t.id IS NULL OR tt.allow_checkin_checkout = '1')");

        query += ` WHERE ${whereConditions.join(" AND ")}`;
        query += ` ORDER BY t.checkin_at DESC, t.start_time DESC`;

        const conveyance = await db.query(query, params);
        return conveyance;
    } catch (error) {
        console.error("Error fetching conveyance by user id:", error);
        throw error;
    }
}


/*static async findAllClaims(startDate = null, endDate = null) {
    try {


        let query = `SELECT * FROM tasks WHERE claim_status = 'Applied'`;
        let params = [];


        if (startDate && endDate) {
            query += ` AND checkin_at BETWEEN ? AND ?`;
            params.push(startDate, endDate);
        }

        query += ` ORDER BY checkin_at DESC, start_time DESC`;

        const claims = await db.query(query, params);
        return claims;

    } catch (error) {
        console.error("Error fetching claims:", error);
        throw error;
    }
}*/
static async findAllClaims(startDate = null, endDate = null) {
    try {
        let query = `
            SELECT t.*, u.name as user_name, u.emp_id, u.mobile,
                   p.project_name, c.company_name
            FROM tasks t
            INNER JOIN users u ON t.user_id = u.id
            LEFT JOIN projects p ON t.project_id = p.id
            LEFT JOIN companies c ON p.customer_id = c.id
            WHERE t.claim_status = 'Applied' AND u.active = 1
        `;
        let params = [];

        if (startDate && endDate) {
            query += ` AND t.checkin_at BETWEEN ? AND ?`;
            params.push(startDate, endDate);
        }

        query += ` ORDER BY t.checkin_at DESC, t.start_time DESC`;

        const claims = await db.query(query, params);
        return claims;

    } catch (error) {
        console.error("Error fetching claims:", error);
        throw error;
    }
}
/*static async datatableClaims({ page = 1, limit = 10, sortBy = 'checkin_at', sortOrder = 'DESC', search = '' }) {
    const allowedSortFields = ['checkin_at', 'task', 'taskid', 'status', 'claim_status', 'claim_amount'];
    const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'checkin_at';
    const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
    const pageNum = parseInt(page);
    const limitNum = parseInt(limit);
    const offset = (pageNum - 1) * limitNum;

    let baseQuery = `SELECT * FROM tasks WHERE claim_status = 'Applied'`;
    let params = [];

    if (search && search.trim()) {
        const likeSearch = `%${search.trim()}%`;
        baseQuery += ` AND (task LIKE ? OR taskid LIKE ? OR status LIKE ? OR claim_status LIKE ? OR claim_amount LIKE ?)`;
        params.push(likeSearch, likeSearch, likeSearch, likeSearch, likeSearch);
    }

    // Get total count
    const countQuery = baseQuery.replace(/^SELECT \* FROM tasks/, 'SELECT COUNT(*) as total FROM tasks');
    const countResult = await db.query(countQuery, params);
    const total = countResult[0]?.total || 0;

    // Add ORDER BY, LIMIT, OFFSET
    baseQuery += ` ORDER BY ${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
    params.push(limitNum, offset);

    const claims = await db.query(baseQuery, params);
    return { claims, total };
}*/
/*static async datatableClaims({ page = 1, limit = 10, sortBy = 'checkin_at', sortOrder = 'DESC', search = '' }) {
    const allowedSortFields = ['checkin_at', 'task', 'taskid', 'status', 'claim_status', 'claim_amount'];
    const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'checkin_at';
    const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
    const pageNum = parseInt(page);
    const limitNum = parseInt(limit);
    const offset = (pageNum - 1) * limitNum;

    let baseQuery = `
        SELECT t.*, u.name as user_name, u.emp_id, u.mobile,
               p.project_name, c.company_name
        FROM tasks t
        INNER JOIN users u ON t.user_id = u.id
        LEFT JOIN projects p ON t.project_id = p.id
        LEFT JOIN companies c ON p.customer_id = c.id
        WHERE t.claim_status = 'Applied' AND u.active = 1
    `;
    let params = [];

    if (search && search.trim()) {
        const likeSearch = `%${search.trim()}%`;
        baseQuery += ` AND (t.task LIKE ? OR t.taskid LIKE ? OR t.status LIKE ? OR t.claim_status LIKE ? OR t.claim_amount LIKE ? OR u.name LIKE ? OR p.project_name LIKE ? OR c.company_name LIKE ?)`;
        params.push(likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch);
    }

    // Get total count
    const countQuery = `SELECT COUNT(*) as total FROM tasks t INNER JOIN users u ON t.user_id = u.id LEFT JOIN projects p ON t.project_id = p.id LEFT JOIN companies c ON p.customer_id = c.id WHERE t.claim_status = 'Applied' AND u.active = 1` + (search && search.trim() ? ` AND (t.task LIKE ? OR t.taskid LIKE ? OR t.status LIKE ? OR t.claim_status LIKE ? OR t.claim_amount LIKE ? OR u.name LIKE ? OR p.project_name LIKE ? OR c.company_name LIKE ?)` : '');
    const countResult = await db.query(countQuery, params);
    const total = countResult[0]?.total || 0;

    // Add ORDER BY, LIMIT, OFFSET
    baseQuery += ` ORDER BY t.${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
    params.push(limitNum, offset);

    const claims = await db.query(baseQuery, params);
    return { claims, total };
}*/
static async datatableClaims({
    page = 1,
    limit = 10,
    sortBy = 'checkin_at',
    sortOrder = 'DESC',
    search = '',
    startDate = null,
    endDate = null
}) {
    const allowedSortFields = ['checkin_at', 'task', 'taskid', 'status', 'claim_status', 'claim_amount'];
    const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'checkin_at';
    const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
    const pageNum = parseInt(page);
    const limitNum = parseInt(limit);
    const offset = (pageNum - 1) * limitNum;

    let baseQuery = `
        SELECT t.*, u.name as user_name, u.emp_id, u.mobile,
               p.project_name, c.company_name
        FROM tasks t
        INNER JOIN users u ON t.user_id = u.id
        LEFT JOIN projects p ON t.project_id = p.id
        LEFT JOIN companies c ON p.customer_id = c.id
        WHERE t.claim_status = 'Applied' AND u.active = 1
    `;
    let params = [];

    if (search && search.trim()) {
        const likeSearch = `%${search.trim()}%`;
        baseQuery += ` AND (
            t.task LIKE ? OR t.taskid LIKE ? OR t.status LIKE ? OR
            t.claim_status LIKE ? OR t.claim_amount LIKE ? OR
            u.name LIKE ? OR p.project_name LIKE ? OR c.company_name LIKE ?
        )`;
        params.push(likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch);
    }

    if (startDate && endDate) {
        baseQuery += ` AND t.checkin_at BETWEEN ? AND ?`;
        params.push(startDate, endDate);
    }

    // Count query
    let countQuery = `
        SELECT COUNT(*) as total
        FROM tasks t
        INNER JOIN users u ON t.user_id = u.id
        LEFT JOIN projects p ON t.project_id = p.id
        LEFT JOIN companies c ON p.customer_id = c.id
        WHERE t.claim_status = 'Applied' AND u.active = 1
    `;
    let countParams = [...params];

    if (search && search.trim()) {
        countQuery += ` AND (
            t.task LIKE ? OR t.taskid LIKE ? OR t.status LIKE ? OR
            t.claim_status LIKE ? OR t.claim_amount LIKE ? OR
            u.name LIKE ? OR p.project_name LIKE ? OR c.company_name LIKE ?
        )`;
    }

    if (startDate && endDate) {
        countQuery += ` AND t.checkin_at BETWEEN ? AND ?`;
    }

    const countResult = await db.query(countQuery, countParams);
    const total = countResult[0]?.total || 0;

    baseQuery += ` ORDER BY t.${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
    params.push(limitNum, offset);

    const claims = await db.query(baseQuery, params);
    return { claims, total };
}
static async getUserById(userId) {
    try {
        const query = `SELECT id, name, branch_id FROM users WHERE id = ? AND active = 1`;
        const [user] = await db.query(query, [userId]);
        return user;
    } catch (error) {
        console.error("Error fetching user by ID:", error);
        throw error;
    }
}

static async findFinanceClaims(startDate = null, endDate = null, userBranchIds = []) {
    try {
        // If no branch IDs provided, return empty array
        if (!userBranchIds || userBranchIds.length === 0) {
            return [];
        }

        let query = `
            SELECT t.*, u.name as user_name, u.branch_id, b.branch_name
            FROM tasks t
            INNER JOIN users u ON t.user_id = u.id
            LEFT JOIN branches b ON FIND_IN_SET(b.id, u.branch_id)
            WHERE t.claim_status = 'Approved'
            AND u.active = 1
            AND (
        `;

        // Build dynamic OR conditions for each finance user's accessible branch ID
        const branchConditions = userBranchIds.map(() => `FIND_IN_SET(?, u.branch_id)`).join(' OR ');
        query += branchConditions + ')';

        let params = [...userBranchIds];

        if (startDate && endDate) {
            query += ` AND t.checkin_at BETWEEN ? AND ?`;
            params.push(startDate, endDate);
        }

        query += ` ORDER BY t.checkin_at DESC, t.start_time DESC`;

        let claims = await db.query(query, params);
        return claims;

    } catch (error) {
        console.error("Error fetching finance claims:", error);
        throw error;
    }
}

static async datatableFinanceClaims({ page = 1, limit = 10, sortBy = 'checkin_at', sortOrder = 'DESC', search = '', userBranchIds = [] }) {
    if (!userBranchIds || userBranchIds.length === 0) {
        return { claims: [], total: 0 };
    }
    const allowedSortFields = ['checkin_at', 'task', 'taskid', 'status', 'claim_status', 'claim_amount'];
    const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'checkin_at';
    const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
    const pageNum = parseInt(page);
    const limitNum = parseInt(limit);
    const offset = (pageNum - 1) * limitNum;

    let baseQuery = `SELECT t.*, u.name as user_name, u.branch_id, b.branch_name FROM tasks t INNER JOIN users u ON t.user_id = u.id LEFT JOIN branches b ON FIND_IN_SET(b.id, u.branch_id) WHERE t.claim_status = 'Approved' AND u.active = 1 AND (`;
    const branchConditions = userBranchIds.map(() => `FIND_IN_SET(?, u.branch_id)`).join(' OR ');
    baseQuery += branchConditions + ')';
    let params = [...userBranchIds];

    if (search && search.trim()) {
        const likeSearch = `%${search.trim()}%`;
        baseQuery += ` AND (t.task LIKE ? OR t.taskid LIKE ? OR t.status LIKE ? OR t.claim_status LIKE ? OR t.claim_amount LIKE ? OR u.name LIKE ? OR b.branch_name LIKE ?)`;
        params.push(likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch, likeSearch);
    }

    // Get total count
    let countBaseQuery = baseQuery;
    let countQuery = `SELECT COUNT(*) as total FROM (${countBaseQuery} GROUP BY t.id) as sub`;
    const countResult = await db.query(countQuery, params);
    const total = countResult[0]?.total || 0;

    // Add ORDER BY, LIMIT, OFFSET
    baseQuery += ` GROUP BY t.id ORDER BY t.${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
    params.push(limitNum, offset);

    const claims = await db.query(baseQuery, params);
    return { claims, total };
}

// update claim status
static async updateClaimStatus(id, status, remarks, approved_by) {
    try {
        console.log(id, status, remarks, approved_by)
        const result = await db.query(`UPDATE tasks SET claim_status = ? , claim_approved_remarks = ? , approved_date = ?, approved_by = ? WHERE id = ?`,
        [status, remarks, new Date(), approved_by, id]);
        return result;
    } catch (error) {
        console.error("Error updating claim status:", error);
        throw error;
    }
}

}