import { db } from "../utils/database.js";

async function findAllConveyance(startDate = null, endDate = null) {
    try {
        let query = `
            SELECT 
                u.id as emp_id,
                u.name,
                COUNT(CASE WHEN t.id IS NOT NULL THEN t.id END) as total_visits,
                COALESCE(SUM(CASE WHEN t.id IS NOT NULL THEN t.distance_travelled ELSE 0 END), 0) as total_distance,
                COALESCE(SUM(CASE WHEN t.id IS NOT NULL THEN t.claim_amount ELSE 0 END), 0) as total_claim_amount,
                COALESCE(SEC_TO_TIME(SUM(CASE WHEN t.id IS NOT NULL THEN t.time_spent ELSE 0 END) * 60), '00:00:00') as total_time_spent_minutes
            FROM users u
            LEFT JOIN tasks t ON u.id = t.user_id 
                AND t.tasktype_id != 1 
                AND t.status = 'Closed'
                AND t.checkout_at != ''
            LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
                AND tt.allow_checkin_checkout = '1'
        `;
        
        let params = [];
        let whereConditions = ["u.active = 1"];

        // Add date filtering if provided (using checkin_at instead of start_date)
        if (startDate && endDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) BETWEEN ? AND ?)");
            params.push(startDate, endDate);
        } else if (startDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) >= ?)");
            params.push(startDate);
        } else if (endDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) <= ?)");
            params.push(endDate);
        }

        // Add tasktype condition to ensure only tasks with allow_checkin_checkout = '1' are included
        whereConditions.push("(t.id IS NULL OR tt.allow_checkin_checkout = '1')");

        query += ` WHERE ${whereConditions.join(" AND ")}`;
        query += ` GROUP BY u.id, u.name ORDER BY u.name`;

        const conveyance = await db.query(query, params);
        return conveyance;
    } catch (error) {
        console.error("Error fetching conveyance:", error);
        throw error;
    }
}

async function findConveyanceByUserId(userId, startDate = null, endDate = null) {
    try {
        let query = `
            SELECT 
                u.id as emp_id,
                u.name,
                t.id as task_id,
                t.task,
                t.start_date,
                t.start_time,
                t.end_time,
                t.distance_travelled as distance,
                t.claim_amount,
                COALESCE(SEC_TO_TIME(t.time_spent * 60), '00:00:00') as time_spent_minutes,
                t.status,
                t.checkin_at,
                t.checkin_location,
                t.checkout_at,
                t.checkout_location
            FROM users u
            LEFT JOIN tasks t ON u.id = t.user_id 
                AND t.tasktype_id in (1, 3) 
                AND t.status = 'Closed'
                AND t.checkout_at != ''
            LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
                AND tt.allow_checkin_checkout = '1'
        `;
        
        let params = [userId];
        let whereConditions = ["u.id = ?", "u.active = 1"];

        // Add date filtering if provided (using checkin_at instead of start_date)
        if (startDate && endDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) BETWEEN ? AND ?)");
            params.push(startDate, endDate);
        } else if (startDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) >= ?)");
            params.push(startDate);
        } else if (endDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) <= ?)");
            params.push(endDate);
        }

        // Add tasktype condition to ensure only tasks with allow_checkin_checkout = '1' are included
        whereConditions.push("(t.id IS NULL OR tt.allow_checkin_checkout = '1')");

        query += ` WHERE ${whereConditions.join(" AND ")}`;
        query += ` ORDER BY t.checkin_at DESC, t.start_time DESC`;

        const conveyance = await db.query(query, params);
        return conveyance;
    } catch (error) {
        console.error("Error fetching conveyance by user id:", error);
        throw error;
    }
}



export {
    findAllConveyance,
    findConveyanceByUserId    
}
