import { db } from "../utils/database.js";

function toNull(val) {
    return val === undefined ? null : val;
}

export async function findConveyanceRolewise({ scope, userId, branchId, startDate, endDate }) {
    try {
        let query = `
            SELECT 
                u.id as id,
                u.emp_id as emp_id,
                u.name,
                COUNT(CASE WHEN t.id IS NOT NULL THEN t.id END) as total_visits,
                COALESCE(SUM(CASE WHEN t.id IS NOT NULL THEN t.distance_travelled ELSE 0 END), 0) as total_distance,
                COALESCE(SUM(CASE WHEN t.id IS NOT NULL THEN t.claim_amount ELSE 0 END), 0) as total_claim_amount,
                COALESCE(SEC_TO_TIME(SUM(t.time_spent))) AS total_time_spent_minutes,
                COALESCE(SEC_TO_TIME(SUM(CASE WHEN t.id IS NOT NULL THEN t.time_spent ELSE 0 END) * 60), '00:00:00') as total_time_spent
            FROM users u
            LEFT JOIN tasks t ON u.id = t.user_id 
                AND t.tasktype_id != 1 
                AND t.status = 'Closed'
                AND t.checkout_at != ''
            LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
                AND tt.allow_checkin_checkout = '1'
        `;

        let params = [];
        let whereConditions = ["u.active = 1"];

        // Role-based filtering
        if (scope === 'owned') {
            whereConditions.push("u.id = ?");
            params.push(toNull(userId));
        } else if (scope === 'branch') {
            whereConditions.push("u.branch_id = ?");
            params.push(toNull(branchId));
        }
        // If global, no extra filter

        // Date filtering (using checkin_at)
        if (startDate && endDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) BETWEEN ? AND ?)");
            params.push(startDate, endDate);
        } else if (startDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) >= ?)");
            params.push(startDate);
        } else if (endDate) {
            whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) <= ?)");
            params.push(endDate);
        }

        // Add tasktype condition to ensure only tasks with allow_checkin_checkout = '1' are included
        whereConditions.push("(t.id IS NULL OR tt.allow_checkin_checkout = '1')");

        query += ` WHERE ${whereConditions.join(" AND ")}`;
        query += ` GROUP BY u.id, u.name ORDER BY u.name`;

        let conveyance = await db.query(query, params);
        return conveyance;
    } catch (error) {
        console.error("Error fetching conveyance rolewise:", error);
        throw error;
    }
}

export class ConveyanceRolewiseModel {
    // static async datatableList({ page = 1, limit = 10, sortBy = 'name', sortOrder = 'ASC', search = '' }) {
    //     const allowedSortFields = ['name', 'emp_id', 'total_visits', 'total_distance', 'total_claim_amount', 'total_time_spent_minutes'];
    //     const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'name';
    //     const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'DESC' ? 'DESC' : 'ASC';
    //     const pageNum = parseInt(page);
    //     const limitNum = parseInt(limit);
    //     const offset = (pageNum - 1) * limitNum;

    //     let baseQuery = `SELECT u.id as id, u.emp_id as emp_id, u.name, COUNT(CASE WHEN t.id IS NOT NULL THEN t.id END) as total_visits, COALESCE(SUM(CASE WHEN t.id IS NOT NULL THEN t.distance_travelled ELSE 0 END), 0) as total_distance, COALESCE(SUM(CASE WHEN t.id IS NOT NULL THEN t.claim_amount ELSE 0 END), 0) as total_claim_amount, COALESCE(SEC_TO_TIME(SUM(t.time_spent))) AS total_time_spent_minutes, FROM users u LEFT JOIN tasks t ON u.id = t.user_id AND t.tasktype_id != 1 AND t.status = 'Closed' AND t.checkout_at != '' LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id AND tt.allow_checkin_checkout = '1'`;
    //     let params = [];
    //     let whereConditions = ["u.active = 1"];

    //     if (search && search.trim()) {
    //         const likeSearch = `%${search.trim()}%`;
    //         whereConditions.push("(u.emp_id LIKE ? OR u.name LIKE ?)");
    //         params.push(likeSearch, likeSearch);
    //     }

    //     // Add tasktype condition to ensure only tasks with allow_checkin_checkout = '1' are included
    //     whereConditions.push("(t.id IS NULL OR tt.allow_checkin_checkout = '1')");

    //     let whereClause = whereConditions.length > 0 ? ` WHERE ${whereConditions.join(" AND ")}` : '';
    //     let groupOrderClause = ` GROUP BY u.id, u.name ORDER BY ${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
    //     let finalQuery = baseQuery + whereClause + groupOrderClause;
    //     params.push(limitNum, offset);

    //     // For total count, use a subquery to avoid 'Unknown column' error
    //     let countBaseQuery = baseQuery + whereClause + ` GROUP BY u.id, u.name`;
    //     let countQuery = `SELECT COUNT(*) as total FROM (${countBaseQuery}) as sub`;
    //     const countResult = await db.query(countQuery, params.slice(0, -2));
    //     const total = countResult[0]?.total || 0;

    //     const conveyance = await db.query(finalQuery, params);
    //     return { conveyance, total };
    // }
    static async datatableList({
    page = 1,
    limit = 10,
    sortBy = 'name',
    sortOrder = 'ASC',
    search = '',
    startDate,
    endDate,
    scope,
    userId,
    branchId
}) {
    const allowedSortFields = [
        'name',
        'emp_id',
        'total_visits',
        'total_distance',
        'total_claim_amount',
        'total_time_spent_minutes'
    ];
    const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'name';
    const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'DESC' ? 'DESC' : 'ASC';
    const pageNum = parseInt(page);
    const limitNum = parseInt(limit);
    const offset = (pageNum - 1) * limitNum;

    let params = [];
    let whereConditions = ["u.active = 1"];

    // 🔍 Role-based filtering
    if (scope === 'owned') {
        whereConditions.push("u.id = ?");
        params.push(toNull(userId));
    } else if (scope === 'branch') {
        whereConditions.push("u.branch_id = ?");
        params.push(toNull(branchId));
    }

    // 🔍 Search filter
    if (search && search.trim()) {
        const likeSearch = `%${search.trim()}%`;
        whereConditions.push("(u.emp_id LIKE ? OR u.name LIKE ?)");
        params.push(likeSearch, likeSearch);
    }
    console.log(">>>>>>>>>>>>>>conveyence filter");
    console.log(startDate);
    console.log(endDate);
    // 📅 Date filtering (checkin_at)
    if (startDate && endDate) {
        whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) BETWEEN ? AND ?)");
        params.push(startDate, endDate);
    } else if (startDate) {
        whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) >= ?)");
        params.push(startDate);
    } else if (endDate) {
        whereConditions.push("(t.id IS NULL OR DATE(t.checkin_at) <= ?)");
        params.push(endDate);
    }

    // ✅ Tasktype condition
    whereConditions.push("(t.id IS NULL OR tt.allow_checkin_checkout = '1')");

    // 🧱 Base query
    let baseQuery = `
        SELECT 
            u.id AS id,
            u.emp_id AS emp_id,
            u.name,
            COUNT(CASE WHEN t.id IS NOT NULL THEN t.id END) AS total_visits,
            COALESCE(SUM(CASE WHEN t.id IS NOT NULL THEN t.distance_travelled ELSE 0 END), 0) AS total_distance,
            COALESCE(SUM(CASE WHEN t.id IS NOT NULL THEN t.claim_amount ELSE 0 END), 0) AS total_claim_amount,
            COALESCE(SEC_TO_TIME(SUM(t.time_spent))) AS total_time_spent_minutes,
            COALESCE(SEC_TO_TIME(SUM(CASE WHEN t.id IS NOT NULL THEN t.time_spent ELSE 0 END) * 60), '00:00:00') AS total_time_spent
        FROM users u
        LEFT JOIN tasks t ON u.id = t.user_id 
            AND t.tasktype_id != 1 
            AND t.status = 'Closed' 
            AND t.checkout_at != ''
        LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id 
            AND tt.allow_checkin_checkout = '1'
    `;

    let whereClause = whereConditions.length > 0 ? ` WHERE ${whereConditions.join(" AND ")}` : '';
    let groupClause = ` GROUP BY u.id, u.emp_id, u.name`;
    let orderClause = ` ORDER BY ${validSortBy} ${validSortOrder}`;
    let paginationClause = ` LIMIT ? OFFSET ?`;

    // 🧮 Total count query
    const countQuery = `
        SELECT COUNT(*) AS total FROM (
            ${baseQuery + whereClause + groupClause}
        ) AS sub
    `;
    const countResult = await db.query(countQuery, params);
    const total = countResult[0]?.total || 0;

    // 📦 Final data query
    const finalQuery = baseQuery + whereClause + groupClause + orderClause + paginationClause;
    const dataParams = [...params, limitNum, offset];
    const conveyance = await db.query(finalQuery, dataParams);

    return { conveyance, total };
}
}