
import { db } from '../utils/database.js';

export const getCustomerById = async (customerId) => {
    const rows = await db.query(
        'SELECT * FROM companies WHERE id = ? AND status = 1',
        [customerId]
    );
    return rows[0] || null;
};

export const getContactsByCustomer = async (customerId) => {
    const rows = await db.query(
        `SELECT c.*, co.company_name
         FROM contacts c
         LEFT JOIN companies co ON c.company_id = co.id
         WHERE c.company_id = ? AND c.status = 1
         ORDER BY c.created_at DESC`,
        [customerId]
    );
    return rows;
};

export const getProjectsByCustomer = async (customerId) => {
    let rows = await db.query(
        'SELECT * FROM projects WHERE customer_id = ?',
        [customerId]
    );

    // If no projects found, return empty array
    if (!rows || rows.length === 0) {
        return [];
    }

    // Get all project IDs for batch queries
    const projectIds = rows.map(project => project.id);
    const projectIdsPlaceholder = projectIds.map(() => '?').join(',');

    // Fetch all project assignees for these projects
    // Fetch all project assignees for these projects
    const assigneesQuery = `
        SELECT pa.project_id, pa.assigned_on, pa.assigned_by, pa.status,
               u.id as user_id, u.name as user_name, u.emp_id, u.email as user_email,
               u.mobile as user_mobile, u.emp_id as user_designation,
               r.name as user_role, r.scope as user_role_scope,
               ab.name as assigned_by_name, ab.emp_id as assigned_by_emp_id,
               ab.email as assigned_by_email, ab.mobile as assigned_by_mobile,
               abr.name as assigned_by_role, abr.scope as assigned_by_role_scope
        FROM projectassignees pa
        LEFT JOIN users u ON pa.user_id = u.id
        LEFT JOIN roles r ON u.roles_id = r.id
        LEFT JOIN users ab ON pa.assigned_by = ab.id
        LEFT JOIN roles abr ON ab.roles_id = abr.id
        WHERE pa.project_id IN (${projectIdsPlaceholder})
        ORDER BY pa.assigned_on DESC
    `;
    const allAssignees = await db.query(assigneesQuery, projectIds);

    // Group assignees by project_id
    const assigneesByProjectId = {};
    allAssignees.forEach(assignee => {
        if (!assigneesByProjectId[assignee.project_id]) {
            assigneesByProjectId[assignee.project_id] = [];
        }
        assigneesByProjectId[assignee.project_id].push({
            user_id: assignee.user_id,
            user_name: assignee.user_name,
            emp_id: assignee.emp_id,
            user_email: assignee.user_email,
            user_mobile: assignee.user_mobile,
            user_designation: assignee.user_designation,
            user_role: assignee.user_role,
            user_role_scope: assignee.user_role_scope,
            assigned_on: assignee.assigned_on,
            assigned_by: assignee.assigned_by,
            assigned_by_name: assignee.assigned_by_name,
            assigned_by_emp_id: assignee.assigned_by_emp_id,
            assigned_by_email: assignee.assigned_by_email,
            assigned_by_mobile: assignee.assigned_by_mobile,
            assigned_by_role: assignee.assigned_by_role,
            assigned_by_role_scope: assignee.assigned_by_role_scope,
            status: assignee.status
        });
    });

    // Merge assignees with projects
    const projectsWithAssignees = rows.map(project => ({
        ...project,
        project_assignees: assigneesByProjectId[project.id] || [],
        assignees_count: (assigneesByProjectId[project.id] || []).length
    }));

    return projectsWithAssignees;
};
export const getTasksByCustomer = async (customerId) => {
    const rows = await db.query(
        `SELECT t.*,
                tt.activity_name as task_type_name,
                p.project_name, p.project_id as project_code,
                c.company_name, c.company_id as customer_code,
                cont.contact_name, cont.email as contact_email,
                u.name as assigned_user_name
         FROM tasks t
         LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
         LEFT JOIN projects p ON t.project_id = p.id
         LEFT JOIN companies c ON p.customer_id = c.id
         LEFT JOIN contacts cont ON t.contact_id = cont.id
         LEFT JOIN users u ON t.user_id = u.id
         WHERE p.customer_id = ? AND t.status != 'Deleted'
         ORDER BY t.created_at DESC`,
        [customerId]
    );

    // If no tasks found, return empty array
    if (!rows || rows.length === 0) {
        return [];
    }

    // Get all task IDs for batch queries
    const taskIds = rows.map(task => task.id);
    const taskIdsPlaceholder = taskIds.map(() => '?').join(',');

    // Fetch all task notes for these tasks
    const notesQuery = `
        SELECT tn.*, u.name as user_name, u.emp_id
        FROM tasknotes tn
        LEFT JOIN users u ON tn.user_id = u.id
        WHERE tn.task_id IN (${taskIdsPlaceholder})
        ORDER BY tn.created_at DESC
    `;
    const allNotes = await db.query(notesQuery, taskIds);

    // Fetch all task uploads for these tasks
    const uploadsQuery = `
        SELECT tu.id as upload_id, tu.task_id, tu.file_name, tu.created_at, tu.updated_at,
               t.taskcategoryoption_id, tc.dropdown as task_category_option
        FROM taskuploads tu
        LEFT JOIN tasks t ON tu.task_id = t.id
        LEFT JOIN taskcategoryoptions tc ON t.taskcategoryoption_id = tc.id
        WHERE tu.task_id IN (${taskIdsPlaceholder})
        ORDER BY tu.created_at DESC
    `;
    const allUploads = await db.query(uploadsQuery, taskIds);

    // Group notes and uploads by task_id
    const notesByTaskId = {};
    const uploadsByTaskId = {};

    allNotes.forEach(note => {
        if (!notesByTaskId[note.task_id]) {
            notesByTaskId[note.task_id] = [];
        }
        notesByTaskId[note.task_id].push(note);
    });

    allUploads.forEach(upload => {
        if (!uploadsByTaskId[upload.task_id]) {
            uploadsByTaskId[upload.task_id] = [];
        }
        uploadsByTaskId[upload.task_id].push(upload);
    });

    // Merge notes and uploads with tasks
    const tasksWithDetails = rows.map(task => ({
        ...task,
        task_notes: notesByTaskId[task.id] || [],
        task_uploads: uploadsByTaskId[task.id] || [],
        notes_count: (notesByTaskId[task.id] || []).length,
        uploads_count: (uploadsByTaskId[task.id] || []).length
    }));

    return tasksWithDetails;
};

export const getFilesByCustomer = async (customerId) => {
    // Placeholder: This would need to be implemented based on your files table structure
    // For now, returning empty array similar to contact model
    return [];
};


