import { db } from '../utils/database.js';

export class Package {
    static async create(packageData, userId) {
        try {
            // Insert package
            const packageQuery = `
                INSERT INTO packages (
                    package_name, status, created_by,
                    created_at, updated_at
                ) VALUES (?, 1, ?, NOW(), NOW())
            `;
            
            const result = await db.query(packageQuery, [
                packageData.package_name,
                userId
            ]);
            
            const packageId = result.insertId;

            // Insert package products with quantity
            if (packageData.products && packageData.products.length > 0) {
                const productQuery = `
                    INSERT INTO packageproducts (
                        package_id, product_id, quantity, created_at, updated_at
                    ) VALUES (?, ?, ?, NOW(), NOW())
                `;
                
                // Each product should be an object: { product_id, quantity }
                for (const prod of packageData.products) {
                    await db.query(productQuery, [packageId, prod.product_id, prod.quantity || 1]);
                }
            }

            return packageId;

        } catch (error) {
            throw error;
        }
    }

    static async update(id, packageData) {
        try {
            // Update package
            const packageQuery = `
                UPDATE packages 
                SET package_name = ?,
                    updated_at = NOW()
                WHERE id = ? AND status = 1
            `;
            
            await db.query(packageQuery, [packageData.package_name, id]);

            // Update package products with quantity
            if (packageData.products) {
                // Remove existing products
                await db.query('DELETE FROM packageproducts WHERE package_id = ?', [id]);

                // Insert new products with quantity
                if (packageData.products.length > 0) {
                    const productQuery = `
                        INSERT INTO packageproducts (
                            package_id, product_id, quantity, created_at, updated_at
                        ) VALUES (?, ?, ?, NOW(), NOW())
                    `;
                    
                    // Each product should be an object: { product_id, quantity }
                    for (const prod of packageData.products) {
                        await db.query(productQuery, [id, prod.product_id, prod.quantity || 1]);
                    }
                }
            }

            return true;

        } catch (error) {
            throw error;
        }
    }

    static async softDelete(id) {
        const query = `
            UPDATE packages 
            SET status = 0,
                updated_at = NOW() 
            WHERE id = ?
        `;
        return db.query(query, [id]);
    }

    static async findById(id) {
        const packageQuery = `
            SELECT p.*, u.name as creator_name 
            FROM packages p
            LEFT JOIN users u ON p.created_by = u.id
            WHERE p.id = ? AND p.status = 1
        `;
        const [package_] = await db.query(packageQuery, [id]);

        if (package_) {
            // Get package products
            const productsQuery = `
                SELECT p.* 
                FROM products p
                INNER JOIN packageproducts pp ON p.id = pp.product_id
                WHERE pp.package_id = ? AND p.status = 1
            `;
            package_.products = await db.query(productsQuery, [id]);
        }

        return package_;
    }

    static async list() {
        const packagesQuery = `
            SELECT p.*, u.name as creator_name 
            FROM packages p
            LEFT JOIN users u ON p.created_by = u.id
            WHERE p.status = 1 
            ORDER BY p.created_at DESC
        `;
        const packages = await db.query(packagesQuery);

        // Get products for each package
        for (let package_ of packages) {
            const productsQuery = `
                SELECT p.*,pp.quantity
                FROM products p
                INNER JOIN packageproducts pp ON p.id = pp.product_id
                WHERE pp.package_id = ? AND p.status = 1
            `;
            package_.products = await db.query(productsQuery, [package_.id]);
        }

        return packages;
    }

    static async getPackageDetails(packageId) {
        try {
            console.log('Getting package details for ID:', packageId);

            const packageQuery = `
                SELECT * FROM packages 
                WHERE id = ? AND status = 1
            `;

            let productsQuery = `
                SELECT 
                    p.*
                FROM packageproducts pp
                LEFT JOIN products p ON pp.product_id = p.id
                WHERE pp.package_id = ? AND p.status = 1
            `;
            console.log('Executing products query:', productsQuery, 'with ID:', packageId);
            // Get package data
            const [packageData] = await db.query(packageQuery, [packageId]);
            //const packageData = packageResults?.[0];

            console.log('Package query result:', packageData);

            if (!packageData) {
                throw new Error('Package not found');
            }

            // Get products with proper array handling
            let rows = await db.query(productsQuery, [packageId]);
            
            // Debug logging
            console.log('Raw query result:', rows);
            console.log('Is Array:', Array.isArray(rows));
            console.log('Number of rows:', rows?.length);

            // Ensure we have an array of products
            const products = Array.isArray(rows) ? rows : [rows].filter(Boolean);

            return {
                package: {
                    id: packageData.id,
                    name: packageData.package_name || packageData.name,
                    description: packageData.description,
                    price: packageData.price,
                    status: packageData.status === 1,
                    created_at: packageData.created_at,
                    updated_at: packageData.updated_at
                },
                products: products.map(product => ({
                    id: product.id,
                    name: product.name,
                    status: product.status === 1,
                    type: product.type,
                    model: product.model,
                    sku_no: product.sku_no,
                    specifications: product.specifications === 1,
                    installation_instructions: product.installation_instructions
                }))
            };
        } catch (error) {
            console.error('Get package details error:', error);
            throw error;
        }
    }

   /* static async getPackageDetails(projectId) {
        try {
            // 1. Get all packages for the given project
            const packageRows = await db.query(
                `SELECT pk.* 
                 FROM projectpackages pp
                 LEFT JOIN packages pk ON pp.package_id = pk.id
                 WHERE pp.project_id = ? AND pk.status = 1`,
                [projectId]
            );

            if (!packageRows || packageRows.length === 0) {
                throw new Error('No packages found for this project');
            }

            // 2. For each package, get its products
            const packages = [];
            for (const packageData of packageRows) {
                const productsQuery = `
                    SELECT p.*, pp.quantity
                    FROM packageproducts pp
                    LEFT JOIN products p ON pp.product_id = p.id
                    WHERE pp.package_id = ? AND p.status = 1
                `;
                const productRows = await db.query(productsQuery, [packageData.id]);

                packages.push({
                    id: packageData.id,
                    name: packageData.package_name || packageData.name,
                    description: packageData.description,
                    price: packageData.price,
                    status: packageData.status === 1,
                    created_at: packageData.created_at,
                    updated_at: packageData.updated_at,
                    products: (productRows || []).map(product => ({
                        id: product.id,
                        name: product.name,
                        status: product.status === 1,
                        type: product.type,
                        model: product.model,
                        sku_no: product.sku_no,
                        specifications: product.specifications,
                        installation_instructions: product.installation_instructions,
                        quantity: product.quantity
                    }))
                });
            }

            return { packages };
        } catch (error) {
            console.error('Get package details error:', error);
            throw error;
        }
    }*/
}