import { db } from '../utils/database.js';
// import Task, Tasktype, etc. as needed for additional calculations

export class Project {
    static async generateProjectId() {
        const query = `
            SELECT project_id 
            FROM projects 
            WHERE project_id LIKE 'EP%' 
            and status = 1 ORDER BY id DESC LIMIT 1
        `;
        const [result] = await db.query(query);
        
        if (!result) {
            return 'EP0001';
        }
        
        const lastId = parseInt(result.project_id.substring(2));
        return `EP${String(lastId + 1).padStart(4, '0')}`;
    }

    static async getBranchIdByCompanyId(companyId) {
        const query = `
            SELECT branch_id 
            FROM companies 
            WHERE id = ? 
            AND status = 1
        `;
        const [company] = await db.query(query, [companyId]);
        return company ? company.branch_id : null;
    }
       static async create(projectData, userId) {
        // Validation for required fields
        const requiredFields = [
            'project_name',
            'customer_id',
            'contact_id',
            'start_date',
            'closure_date',
            'project_latitude',
            'project_longitude',
            'address',
            'project_manager'
        ];
        const missingFields = requiredFields.filter(field => !projectData[field]);
        if(missingFields.length > 0) {
            throw new Error(`Missing required fields: ${missingFields.join(', ')}`);
        }
        try {
            const projectId = await this.generateProjectId();
            // Fetch branch_id from company
            const branchId = await this.getBranchIdByCompanyId(projectData.customer_id);
            if (!branchId) {
                throw new Error('Branch not found for the given company');
            }
            // Handle empty date values - convert empty strings to null
            const hoServiceDate = projectData.ho_service_date && projectData.ho_service_date.trim() !== '' 
                ? projectData.ho_service_date 
                : null;

            const query = `
                INSERT INTO projects (
                    project_id,project_name, customer_id, customer_branch_id,
                    contact_id,
                    start_date, closure_date, project_latitude, project_longitude,
                    address, notes, payment_terms, ho_service_date,
                    project_manager, customer_po_value, created_by,
                    status, created_at, updated_at
                ) VALUES (
                    ?,?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1, NOW(), NOW()
                )
            `;

            // Calculate total po_value from all packages
            let povalues = 0;
            if (Array.isArray(projectData.packages)) {
                povalues = projectData.packages.reduce((sum, pkg) => sum + Number(pkg.po_value || 0), 0);
            }
            const result = await db.query(query, [
                projectId,
                projectData.project_name,
                projectData.customer_id,
                branchId,
                projectData.contact_id,
                projectData.start_date,
                projectData.closure_date,
                projectData.project_latitude,
                projectData.project_longitude,
                projectData.address,
                projectData.notes,
                projectData.payment_terms,
                hoServiceDate,
                projectData.project_manager,
                povalues,
                userId
            ]);
            const newProjectId = result.insertId;

            // Insert into projectpackages table for each package
            if (Array.isArray(projectData.packages)) {
                for (const pkg of projectData.packages) {
                    await db.query(
                        `INSERT INTO projectpackages 
                            (project_id, customer_po, po_value, package_id, created_date, created_by, created_at, updated_at)
                         VALUES (?, ?, ?, ?, CURDATE(), ?, NOW(), NOW())`,
                        [
                            newProjectId,
                            pkg.customer_po,
                            pkg.po_value,
                            pkg.package_id,
                            userId
                        ]
                    );
                }
            }

            // After creating the project and getting projectId
            // projectManagerId: the user_id of the project manager
            // userId: the logged-in user's id (who is assigning)

            await db.query(
                `INSERT INTO projectassignees 
                    (project_id, user_id, assigned_on, assigned_by, status, created_at, updated_at)
                 VALUES (?, ?, NOW(), ?, 1, NOW(), NOW())`,
                [newProjectId, projectData.project_manager, userId]
            );

            return newProjectId;
        } catch (error) {
            console.error('Create project error:', error);
            throw error;
        }
    }
    /*static async create(projectData, userId) {
        // Validation for required fields
        const requiredFields = [
            'project_name',
            'customer_id',
            'contact_id',
            'start_date',
            'closure_date',
            'project_latitude',
            'project_longitude',
            'address',
            'project_manager'
        ];
        const missingFields = requiredFields.filter(field => !projectData[field]);
        if (missingFields.length > 0) {
            throw new Error(`Missing required fields: ${missingFields.join(', ')}`);
        }

        try {
            const projectId = await this.generateProjectId();

            // Fetch branch_id from company
            const branchId = await this.getBranchIdByCompanyId(projectData.customer_id);
            if (!branchId) {
                throw new Error('Branch not found for the given company');
            }

            // Add contact_id to the insert query and values
            const query = `
                INSERT INTO projects (
                    project_id,project_name, customer_id, customer_branch_id,
                    contact_id,
                    start_date, closure_date, project_latitude, project_longitude,
                    address, notes, payment_terms, ho_service_date,
                    project_manager, customer_po_value, created_by,
                    status, created_at, updated_at
                ) VALUES (
                    ?,?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1, NOW(), NOW()
                )
            `;

            // Calculate total po_value from all packages
            let povalues = 0;
            if (Array.isArray(projectData.packages)) {
                povalues = projectData.packages.reduce((sum, pkg) => sum + Number(pkg.po_value || 0), 0);
            }

            const result = await db.query(query, [
                projectId,
                projectData.project_name,
                projectData.customer_id,
                branchId,
                projectData.contact_id, // <-- added here
                projectData.start_date,
                projectData.closure_date,
                projectData.project_latitude,
                projectData.project_longitude,
                projectData.address,
                projectData.notes,
                projectData.payment_terms,
                projectData.ho_service_date,
                projectData.project_manager,
                povalues,
                userId
            ]);
            const newProjectId = result.insertId;

            // Insert into projectpackages table for each package
            if (Array.isArray(projectData.packages)) {
                for (const pkg of projectData.packages) {
                    await db.query(
                        `INSERT INTO projectpackages 
                            (project_id, customer_po, po_value, package_id, created_date, created_by, created_at, updated_at)
                         VALUES (?, ?, ?, ?, CURDATE(), ?, NOW(), NOW())`,
                        [
                            newProjectId,
                            pkg.customer_po,
                            pkg.po_value,
                            pkg.package_id,
                            userId
                        ]
                    );
                }
            }

            // After creating the project and getting projectId
            // projectManagerId: the user_id of the project manager
            // userId: the logged-in user's id (who is assigning)

            await db.query(
                `INSERT INTO projectassignees 
                    (project_id, user_id, assigned_on, assigned_by, status, created_at, updated_at)
                 VALUES (?, ?, NOW(), ?, 1, NOW(), NOW())`,
                [newProjectId, projectData.project_manager, userId]
            );

            return newProjectId;
        } catch (error) {
            console.error('Create project error:', error);
            throw error;
        }
    }*/

    static async update(id, projectData) {
        // Calculate total po_value from all packages
        let povalues = 0;
        if (Array.isArray(projectData.packages)) {
            povalues = projectData.packages.reduce((sum, pkg) => sum + Number(pkg.po_value || 0), 0);
        }

        // Add contact_id to the update query and values
        const query = `
            UPDATE projects 
            SET project_name = ?,
                customer_id = ?,
                customer_branch_id = ?,
                contact_id = ?,
                start_date = ?,
                closure_date = ?,
                project_latitude = ?,
                project_longitude = ?,
                address = ?,
                notes = ?,
                payment_terms = ?,
                ho_service_date = ?,
                project_manager = ?,
                updated_at = NOW()
            WHERE id = ? AND status = 1
        `;
        // Handle empty date values - convert empty strings to null
        const hoServiceDate = projectData.ho_service_date && projectData.ho_service_date.trim() !== '' 
            ? projectData.ho_service_date 
            : null;

        await db.query(query, [
            projectData.project_name,
            projectData.customer_id,
            projectData.customer_branch_id ?? null,
            projectData.contact_id, // <-- added here
            projectData.start_date,
            projectData.closure_date,
            projectData.project_latitude,
            projectData.project_longitude,
            projectData.address,
            projectData.notes,
            projectData.payment_terms,
            hoServiceDate,
            projectData.project_manager,
            id
        ]);

        // Remove existing projectpackages for this project
        await db.query('DELETE FROM projectpackages WHERE project_id = ?', [id]);

        // Insert new projectpackages
        if (Array.isArray(projectData.packages)) {
            for (const pkg of projectData.packages) {
                await db.query(
                    `INSERT INTO projectpackages 
                        (project_id, customer_po, po_value, package_id, created_date, created_by, created_at, updated_at)
                     VALUES (?, ?, ?, ?, CURDATE(), ?, NOW(), NOW())`,
                    [
                        id,
                        pkg.customer_po,
                        pkg.po_value,
                        pkg.package_id,
                        projectData.updated_by || 0 // or userId if available
                    ]
                );
            }
        }

        return true;
    }

    // When fetching products for a package, include product_name:
    static async getProductsForPackage(packageId) {
        const query = `
            SELECT pr.*, pr.name as product_name
            FROM packageproducts pp
            INNER JOIN products pr ON pp.product_id = pr.id
            WHERE pp.package_id = ? AND pr.status = 1
        `;
        const products = await db.query(query, [packageId]);
        return products;
    }

    static async softDelete(id) {
        const query = `
            UPDATE projects 
            SET status = 0,
                updated_at = NOW() 
            WHERE id = ?
        `;
        return db.query(query, [id]);
    }

    static async findById(id) {
        const query = `
            SELECT p.*, 
                   c.company_name as customer_name,
                   u.name as manager_name,
                   pk.package_name,
                   pk.id as package_id
            FROM projects p
            LEFT JOIN companies c ON p.customer_id = c.id
            LEFT JOIN users u ON p.project_manager = u.id
            LEFT JOIN packages pk ON p.package = pk.id
            WHERE p.id = ? AND p.status = 1
        `;
        const [project] = await db.query(query, [id]);
        return project;
    }

    static async list(userId, start_date, end_date) {
        if (!userId) throw new Error('userId is required');
        let users = await db.query(
            `SELECT branch_id, roles_id FROM users WHERE id = ? AND active = 1`,
            [userId]
        );
        const user = users && users[0] ? users[0] : null;
        if (!user) throw new Error('User not found');

        let roles = await db.query(
            `SELECT access_scope FROM roles WHERE id = ? AND active = 1`,
            [user.roles_id]
        );
        const role = roles && roles[0] ? roles[0] : null;
        if (!role) throw new Error('Role not found');

        const access_scope = role.access_scope;
        const branch_id = user.branch_id;

        let query = `
            SELECT p.*, 
                   c.company_name as customer_name,
                   u.name as manager_name,
                   pk.package_name,
                   pk.id as package_id
            FROM projects p
            LEFT JOIN companies c ON p.customer_id = c.id
            LEFT JOIN users u ON p.project_manager = u.id
            LEFT JOIN packages pk ON p.package = pk.id
            WHERE p.status = 1
        `;
        const params = [];

        if (access_scope === 'owned') {
            query += `
                AND EXISTS (
                    SELECT 1 FROM projectassignees pa
                    WHERE pa.project_id = p.id AND pa.user_id = ?
                )
            `;
            params.push(userId);
        } else if (access_scope === 'branch') {
            const branchIds = (branch_id || '')
                .split(',')
                .map(id => id.trim())
                .filter(id => id.length > 0);

            if (branchIds.length > 0) {
                query += ` AND p.customer_branch_id IN (${branchIds.map(() => '?').join(',')}) `;
                params.push(...branchIds);
            }
        }
        // If global, no extra filter

        // Add date filter
        if (start_date && end_date) {
            query += ` AND DATE(p.created_at) BETWEEN ? AND ? `;
            params.push(start_date, end_date);
        } else if (start_date) {
            query += ` AND DATE(p.created_at) = ? `;
            params.push(start_date);
        } else if (end_date) {
            query += ` AND DATE(p.created_at) = ? `;
            params.push(end_date);
        }

        query += ` ORDER BY p.created_at DESC`;

        return db.query(query, params);
    }

    static async getAssignedProjects(userId, filterParams = {}) {
        try {
            let query = `
                SELECT p.*,
                       c.company_name, c.company_id as customer_code, 
                       c.company_address, c.email as company_email,
                       c.mobile_no as company_mobile,
                       cont.contact_name, cont.email as contact_email,
                       cont.mobile_no as contact_mobile,
                       pkg.package_name,
                       b.branch_name, b.location as branch_location,
                       b.branch_lat, b.branch_long, b.radius as branch_radius
                FROM projects p
                INNER JOIN projectassignees pa ON p.id = pa.project_id
                LEFT JOIN companies c ON p.customer_id = c.id
                LEFT JOIN contacts cont ON p.contact_id = cont.id
                LEFT JOIN packages pkg ON p.package = pkg.id
                LEFT JOIN branches b ON p.customer_branch_id = b.id
                WHERE pa.user_id = ?
                AND pa.status = 1
                AND p.status = 1
            `;

            const params = [userId];

            // Handle filters
            console.log(filterParams.filter);
            if (filterParams.filter && filterParams.filter_by) {
                switch (filterParams.filter_by) {
                    case 'contacts':
                        query += ` AND p.contact_id = ?`;
                        params.push(filterParams.filter);
                        break;
                    case 'companies':
                        query += ` AND p.customer_id = ?`;
                        params.push(filterParams.filter);
                        break;
                    case 'status':
                        query += ` AND p.project_status = ?`;
                        params.push(filterParams.filter);
                        break;
                    case 'assigned_date':
                        query += ` AND DATE(pa.assigned_on) = ?`;
                        params.push(filterParams.filter);
                        break;
                }
            }

            // Add date range filters if provided
            if (filterParams.start_date) {
                query += ` AND DATE(p.created_at) >= ?`;
                params.push(filterParams.start_date);
            }

            if (filterParams.end_date) {
                query += ` AND DATE(p.created_at) <= ?`;
                params.push(filterParams.end_date);
            }

            query += ` ORDER BY p.created_at DESC`;

            //console.log('Assigned Projects Query:', query, 'Params:', params);
            
            // Fetch projects
            let rows = await db.query(query, params);
            //console.log(rows);

            // Fetch all tasktypes
            let taskTypes = await db.query('SELECT * FROM tasktypes WHERE id!=1 and status = 1');

            // Map projects as before
            const projects = Array.isArray(rows) ? rows.map(project => ({
                id: project.id,
                project_id: project.project_id,
                project_name: project.project_name,
                customer_po: project.customer_po,
                customer_po_value: project.customer_po_value,
                start_date: project.start_date,
                closure_date: project.closure_date,
                project_status: project.project_status,
                customer: {
                    id: project.customer_id,
                    company_name: project.company_name,
                    company_id: project.customer_code,
                    address: project.company_address,
                    email: project.company_email,
                    mobile_no: project.company_mobile
                },
                contact: {
                    id: project.contact_id,
                    name: project.contact_name,
                    email: project.contact_email,
                    mobile_no: project.contact_mobile
                },
                branch: {
                    id: project.customer_branch_id,
                    name: project.branch_name,
                    location: project.branch_location,
                    latitude: project.branch_lat,
                    longitude: project.branch_long,
                    radius: project.branch_radius
                },
                package: {
                    id: project.package,
                    name: project.package_name
                },
                created_at: project.created_at,
                updated_at: project.updated_at
            })) : [];

            return {
                projects,
                tasktypes: Array.isArray(taskTypes) ? taskTypes : []
            };
        } catch (error) {
            console.error('Get assigned projects error:', error);
            throw error;
        }
    }

    static async getProjectDetailsWithTasks(projectId, userId) {
        try {
            // Add debug logging
            console.log('Fetching project details for:', { projectId, userId });

            // Get project details with relations
            const projectQuery = `
                SELECT p.*,
                       c.company_name, c.company_id as customer_code, 
                       c.company_address, c.email as company_email,
                       c.mobile_no as company_mobile,
                       cont.contact_name, cont.email as contact_email,
                       cont.mobile_no as contact_mobile,
                       pkg.package_name,
                       b.branch_name, b.location as branch_location,
                       b.branch_lat, b.branch_long, b.radius as branch_radius
                FROM projects p
                LEFT JOIN companies c ON p.customer_id = c.id
                LEFT JOIN contacts cont ON p.contact_id = cont.id
                LEFT JOIN packages pkg ON p.package = pkg.id
                LEFT JOIN branches b ON p.customer_branch_id = b.id
                WHERE p.id = ? 
                AND p.status = 1
                LIMIT 1
            `;

            let projectArr = await db.query(projectQuery, [projectId]);
            const project = projectArr && projectArr[0] ? projectArr[0] : null;

            if (!project) {
                throw new Error('Project not found');
            }

            // Fetch products for the selected package using packageproducts table
            /*let products = [];
            if (project.package) {
                let productRows = await db.query(
                    `SELECT pr.* FROM packageproducts pp
                     INNER JOIN products pr ON pp.product_id = pr.id
                     WHERE pp.package_id = ? AND pr.status = 1`,
                    [project.package]
                );
                products = productRows || [];
            }*/
            let products = [];
        let packageList = [];
        // Fetch products for the selected package using packageproducts table
        const packageRows = await db.query(
        `SELECT pk.*, pp.customer_po, pp.po_value, pp.package_id
         FROM projectpackages pp
         LEFT JOIN packages pk ON pp.package_id = pk.id
         WHERE pp.project_id = ?`,
        [projectId]
    );

    // For each package, get its products
    for (const pkg of packageRows) {
        const productRows = await db.query(
            `SELECT pr.*, pp.quantity
             FROM packageproducts pp
             INNER JOIN products pr ON pp.product_id = pr.id
             WHERE pp.package_id = ? AND pr.status = 1`,
            [pkg.id]
        );
        products.push(...productRows); // accumulates products instead of overwriting
    }

    packageList = packageRows; // assign fetched packages

            // Tasks query remains the same
            const tasksQuery = `
                SELECT t.*, tt.activity_name as task_type_name,tt.activity_icon as icon FROM tasks t
                LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
                WHERE t.project_id = ? 
                AND t.user_id = ? 
                ORDER BY t.created_at DESC
            `;

            // Get all task types
            const taskTypesQuery = `
                SELECT * FROM tasktypes WHERE id!=1 and status = 1
            `;

            // Execute queries and handle results
           
            let tasksResult=await db.query(tasksQuery, [projectId, userId]);
            let taskTypesResult=await db.query(taskTypesQuery);

            // Extract arrays from results and handle single object case
            const tasks = tasksResult || [];
            const taskTypes = Array.isArray(taskTypesResult[0]) 
                ? taskTypesResult[0] 
                : taskTypesResult[0] ? [taskTypesResult[0]] : [];

            // Debug logging
            console.log('Tasks array:', tasks);
            console.log('Task Types array:', taskTypes);

            // Format and return response
            return {
                project: {
                    id: project.id,
                    project_name: project.project_name,
                    project_id: project.project_id,
                    customer_po: project.customer_po,
                    customer_po_value: project.customer_po_value,
                    start_date: project.start_date,
                    closure_date: project.closure_date,
                    project_status: project.project_status,
                    project_latitude: project.project_latitude,
                    project_longitude: project.project_longitude,
                    project_radius: project.radius,
                    customer: {
                        id: project.customer_id,
                        company_name: project.company_name,
                        company_id: project.customer_code,
                        address: project.company_address,
                        email: project.company_email,
                        mobile_no: project.company_mobile
                    },
                    contact: {
                        id: project.contact_id,
                        name: project.contact_name,
                        email: project.contact_email,
                        mobile_no: project.contact_mobile
                    },
                    branch: {
                        id: project.customer_branch_id,
                        name: project.branch_name,
                        location: project.branch_location,
                        latitude: project.branch_lat,
                        longitude: project.branch_long,
                        radius: project.branch_radius
                    },
                    packageList,
                    products, // <-- Added products array here
                    created_at: project.created_at,
                    updated_at: project.updated_at
                },
                tasks: tasks.map(task => ({
                    id: task.id,
                    title: task.task,
                    description: task.description,
                    start_date: task.start_date,
                    start_time: task.start_time,
                    end_date: task.end_date,
                    end_time: task.end_time,
                    status: task.status,
                    task_type: {
                        id: task.tasktype_id,
                        name: task.task_type_name,
                        icon: task.icon
                    },
                    created_at: task.created_at,
                    updated_at: task.updated_at
                })),
                taskTypes: taskTypes.map(type => ({
                    id: type.id,
                    name: type.activity_name,
                    icon: type.activity_icon,
                    allowCheckinCheckout: type.allow_checkin_checkout === '1',
                    allowStartDateEndDate: type.allow_startdate_enddate === '1',
                    showCalendar: type.show_calendar === '1',
                    markCompleted: type.mark_completed === '1',
                    allowClaim: type.allow_claim === '1',
                    allowEdit: type.allow_edit === '1',
                    totalQuestions: type.totalquestion,
                    status: type.status
                }))
            };
        } catch (error) {
            console.error('Get project details error:', error);
            throw error;
        }
    }
    static async getProjectDetailsWithAdminTasks(projectId, userId) {
         try {
        // Add debug logging
        console.log('Fetching project details for:', { projectId, userId });

        // Get project details with relations
        const projectQuery = `
            SELECT p.*,
                   c.company_name, c.company_id as customer_code, 
                   c.company_address, c.email as company_email,
                   c.mobile_no as company_mobile,
                   cont.contact_name, cont.email as contact_email,
                   cont.mobile_no as contact_mobile,
                   pkg.package_name,
                   b.branch_name, b.location as branch_location,
                   b.branch_lat, b.branch_long, b.radius as branch_radius,u.name as project_manager_name 
            FROM projects p
            LEFT JOIN companies c ON p.customer_id = c.id
            LEFT JOIN contacts cont ON p.contact_id = cont.id
            LEFT JOIN packages pkg ON p.package = pkg.id
            LEFT JOIN branches b ON p.customer_branch_id = b.id
            LEFT JOIN users u ON p.project_manager = u.id
            WHERE p.id = ? 
            AND p.status = 1
            LIMIT 1
        `;

        let projectArr = await db.query(projectQuery, [projectId]);
        const project = projectArr && projectArr[0] ? projectArr[0] : null;

        if (!project) {
            throw new Error('Project not found');
        }

        // Fetch products for the selected package using packageproducts table
        /*let products = [];
        if (project.package) {
            let productRows = await db.query(
                `SELECT pr.* FROM packageproducts pp
                 INNER JOIN products pr ON pp.product_id = pr.id
                 WHERE pp.package_id = ? AND pr.status = 1`,
                [project.package]
            );
            products = productRows || [];
        }*/
        let products = [];
        let packageList = [];
        const packageRows = await db.query(
        `SELECT pk.*, pp.customer_po, pp.po_value, pp.package_id
         FROM projectpackages pp
         LEFT JOIN packages pk ON pp.package_id = pk.id
         WHERE pp.project_id = ?`,
        [projectId]
    );

    // For each package, get its products
    for (const pkg of packageRows) {
        const productRows = await db.query(
            `SELECT pr.*, pp.quantity
             FROM packageproducts pp
             INNER JOIN products pr ON pp.product_id = pr.id
             WHERE pp.package_id = ? AND pr.status = 1`,
            [pkg.id]
        );
        products.push(...productRows); // accumulates products instead of overwriting
    }

    packageList = packageRows; // assign fetched packages

        // Tasks query remains the same
        const tasksQuery = `
            SELECT t.*, tt.activity_name as task_type_name FROM tasks t
            LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
            WHERE t.project_id = ? 
            AND t.user_id = ? 
            ORDER BY t.created_at DESC
        `;

        // Get all task types
        const taskTypesQuery = `
            SELECT * FROM tasktypes WHERE id!=1 and status = 1
        `;

        // --- Additional counts and sums ---
        // Site visit count (tasktype_id = 3)
        let siteVisitCountRow = await db.query(
            `SELECT COUNT(*) as count FROM tasks WHERE project_id = ? AND tasktype_id = 3`,
            [projectId]
        );
        const sitevisit_count = siteVisitCountRow && siteVisitCountRow[0] ? siteVisitCountRow[0].count : 0;

        // Internal notes count (tasktype_id = 6)
        let internalNotesCountRow = await db.query(
            `SELECT COUNT(*) as count FROM tasks WHERE project_id = ? AND tasktype_id = 6`,
            [projectId]
        );
        const internal_notes_count = internalNotesCountRow && internalNotesCountRow[0] ? internalNotesCountRow[0].count : 0;

        // Timespent total for this project
        let timespentRow = await db.query(
            `SELECT SUM(time_spent) as total_timespent FROM tasks WHERE project_id = ?`,
            [projectId]
        );
        const timespent_total = timespentRow && timespentRow[0] && timespentRow[0].total_timespent ? Number(timespentRow[0].total_timespent) : 0;

        // Distance travelled total for this project
        let distanceRow = await db.query(
            `SELECT SUM(distance_travelled) as total_distance FROM tasks WHERE project_id = ?`,
            [projectId]
        );
        const distance_travelled_total = distanceRow && distanceRow[0] && distanceRow[0].total_distance ? Number(distanceRow[0].total_distance) : 0;

        // Execute queries and handle results
        let tasksResult = await db.query(tasksQuery, [projectId, userId]);
        let taskTypesResult = await db.query(taskTypesQuery);

        // Extract arrays from results and handle single object case
        const tasks = tasksResult || [];
        const taskTypes = Array.isArray(taskTypesResult[0]) 
            ? taskTypesResult[0] 
            : taskTypesResult[0] ? [taskTypesResult[0]] : [];
        // Fetch all notes for this project (all tasks)
        const notesQuery = `
            SELECT n.*, t.task 
            FROM tasknotes n
            LEFT JOIN tasks t ON n.task_id = t.id
            WHERE n.project_id = ?
            ORDER BY n.created_at DESC
        `;
        let notesResult = await db.query(notesQuery, [projectId]);
        const notes = notesResult || [];

        // Fetch all files for this project from tasks table (assuming file_name column)
        const filesQuery = `
            SELECT tu.id as upload_id, tu.task_id, tu.file_name as file_name, tu.created_at, tu.updated_at, 
                   t.taskcategoryoption_id,tc.dropdown as task_category_option
            FROM taskuploads tu
            LEFT JOIN tasks t ON tu.task_id = t.id
            LEFT JOIN taskcategoryoptions tc ON t.taskcategoryoption_id = tc.id
            WHERE t.project_id = ?
            ORDER BY tu.created_at DESC
        `;
        let filesResult = await db.query(filesQuery, [projectId]);
        const files = filesResult || [];

        // Fetch assigned users from projectassignees table
        const assigneesQuery = `
            SELECT pa.user_id, u.name, u.emp_id, r.name as role_name
            FROM projectassignees pa
            LEFT JOIN users u ON pa.user_id = u.id
            LEFT JOIN roles r ON u.roles_id = r.id
            WHERE pa.project_id = ? AND pa.status = 1
        `;
        let assigneesResult = await db.query(assigneesQuery, [projectId]);
        const assigned_users = assigneesResult || [];


        // Fetch warranties associated with this project
        const warrantiesQuery = `
            SELECT w.*, pr.name as product_name
            FROM projectwarranties w
            LEFT JOIN products pr ON w.product_id = pr.id
            WHERE w.project_id = ?
            ORDER BY w.created_at DESC
        `;
        let warrantiesResult = await db.query(warrantiesQuery, [projectId]);
        const warranties = warrantiesResult || [];
        let statuses = await db.query(
                `SELECT * FROM projectstatuses WHERE status = 1`
            );
        // Format and return response
        return {
            project: {
                id: project.id,
                project_id: project.project_id,
                project_name: project.project_name,
                customer_po: project.customer_po,
                customer_po_value: project.customer_po_value,
                start_date: project.start_date,
                closure_date: project.closure_date,
                project_status: project.project_status,
                project_manager_name: project.project_manager_name,
                address: project.address,
                notes: project.notes,
                payment_terms: project.payment_terms,
                ho_service_date: project.ho_service_date,
                project_latitude: project.project_latitude,
                project_longitude: project.project_longitude,
                customer: {
                    id: project.customer_id,
                    company_name: project.company_name,
                    company_id: project.customer_code,
                    address: project.company_address,
                    email: project.company_email,
                    mobile_no: project.company_mobile
                },
                contact: {
                    id: project.contact_id,
                    name: project.contact_name,
                    email: project.contact_email,
                    mobile_no: project.contact_mobile
                },
                branch: {
                    id: project.customer_branch_id,
                    name: project.branch_name,
                    location: project.branch_location,
                    latitude: project.branch_lat,
                    longitude: project.branch_long,
                    radius: project.branch_radius
                },
                packageList,
                products,
                assigned_users,
                sitevisit_count,
                internal_notes_count,
                timespent_total,
                distance_travelled_total,
                warranties, // <-- Added warranties array here
                created_at: project.created_at,
                updated_at: project.updated_at
            },
            statuses,
            tasks: tasks.map(task => ({
                id: task.id,
                title: task.task,
                description: task.description,
                start_date: task.start_date,
                start_time: task.start_time,
                end_date: task.end_date,
                end_time: task.end_time,
                status: task.status,
                task_type: {
                    id: task.tasktype_id,
                    name: task.task_type_name
                },
                created_at: task.created_at,
                updated_at: task.updated_at
            })),
            notes: notes.map(note => ({
                id: note.id,
                project_id: note.project_id,
                task_id: note.task_id,
                task: note.task,
                user_id: note.user_id,
                notes: note.notes,
                file_name: note.file_name,
                file: note.file,
                create_date: note.create_date,
                created_at: note.created_at,
                updated_at: note.updated_at
            })),
            files,
            taskTypes: taskTypes.map(type => ({
                id: type.id,
                name: type.activity_name,
                icon: type.activity_icon,
                allowCheckinCheckout: type.allow_checkin_checkout === '1',
                allowStartDateEndDate: type.allow_startdate_enddate === '1',
                showCalendar: type.show_calendar === '1',
                markCompleted: type.mark_completed === '1',
                allowClaim: type.allow_claim === '1',
                allowEdit: type.allow_edit === '1',
                totalQuestions: type.totalquestion,
                status: type.status
            }))
        };
    } catch (error) {
        console.error('Get project details error:', error);
        throw error;
    }
    }

    static async projectFilterList(userId) {
        try {
            // 1. Fetch all assigned projects for the user with company and contact info
            let projects = await db.query(
                `SELECT 
                    p.id as project_id,p.project_name as project_name,
                    c.id as company_id, c.company_name, c.company_address, c.email as company_email, c.mobile_no as company_mobile,
                    cont.id as contact_id, cont.contact_name as contact_name, cont.email as contact_email, cont.mobile_no as contact_mobile
                FROM projects p
                INNER JOIN projectassignees pa ON p.id = pa.project_id
                LEFT JOIN companies c ON p.customer_id = c.id
                LEFT JOIN contacts cont ON p.contact_id = cont.id
                WHERE pa.user_id = ? AND pa.status = 1 AND p.status = 1`,
                [userId]
            );

            if (!projects || projects.length === 0) {
                return { notExists: true };
            }

            // 2. Unique companies
            const companyMap = {};
            const companies = [];
            for (const p of projects) {
                if (p.company_id && !companyMap[p.company_id]) {
                    companyMap[p.company_id] = true;
                    companies.push({
                        id: p.company_id,
                        company_name: p.company_name,
                        company_address: p.company_address,
                        email: p.company_email,
                        mobile_no: p.company_mobile
                    });
                }
            }

            // 3. Unique contacts
            const contactMap = {};
            const contacts = [];
            for (const p of projects) {
                if (p.contact_id && !contactMap[p.contact_id]) {
                    contactMap[p.contact_id] = true;
                    contacts.push({
                        id: p.contact_id,
                        contact_name: p.contact_name,
                        email: p.contact_email,
                        mobile_no: p.contact_mobile
                    });
                }
            }

            // 4. Fetch all statuses from projectstatuses table
            let statuses = await db.query(
                `SELECT * FROM projectstatuses WHERE status = 1`
            );

            return {
                companies,
                contacts,
                statuses: statuses || []
            };
        } catch (error) {
            console.error('Project filter list error:', error);
            throw error;
        }
    }

    static async deleteWarrantyById(id) {
        const query = `DELETE FROM projectwarranties WHERE id = ?`;
        let result = await db.query(query, [id]);
        return result;
    }

    static async getProjectWithPackages(projectId) {
        // Get project details
        const projectRows = await db.query(
            `SELECT p.*, c.company_name, u.name as manager_name
             FROM projects p
             LEFT JOIN companies c ON p.customer_id = c.id
             LEFT JOIN users u ON p.project_manager = u.id
             WHERE p.id = ? AND p.status = 1
             LIMIT 1`,
            [projectId]
        );
        const project = projectRows[0];
        if (!project) throw new Error('Project not found');

        // Get all packages for this project
        const packageRows = await db.query(
            `SELECT pk.*, pp.customer_po, pp.po_value,pp.package_id
             FROM projectpackages pp
             LEFT JOIN packages pk ON pp.package_id = pk.id
             WHERE pp.project_id = ?`,
            [projectId]
        );

        // For each package, get its products
        for (const pkg of packageRows) {
            const productRows = await db.query(
                `SELECT pr.*, pp.quantity
                 FROM packageproducts pp
                 INNER JOIN products pr ON pp.product_id = pr.id
                 WHERE pp.package_id = ? AND pr.status = 1`,
                [pkg.id]
            );
            pkg.products = productRows;
        }

        project.packages = packageRows;
        return project;
    }

    static async datatableList({ page = 1, limit = 10, sortBy = 'created_at', sortOrder = 'DESC', search = '', start_date, end_date }) {
        const allowedSortFields = ['created_at', 'project_name', 'project_id', 'status'];
        const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'created_at';
        const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
        const pageNum = parseInt(page);
        const limitNum = parseInt(limit);
        const offset = (pageNum - 1) * limitNum;

        let baseQuery = `SELECT p.*, c.company_name, u.name as manager_name FROM projects p LEFT JOIN companies c ON p.customer_id = c.id LEFT JOIN users u ON p.project_manager = u.id WHERE 1=1`;
        let params = [];

        // Date filter
        if (start_date && end_date) {
            baseQuery += ` AND DATE(p.created_at) BETWEEN ? AND ? `;
            params.push(start_date, end_date);
        } else if (start_date) {
            baseQuery += ` AND DATE(p.created_at) = ? `;
            params.push(start_date);
        } else if (end_date) {
            baseQuery += ` AND DATE(p.created_at) = ? `;
            params.push(end_date);
        }

        if (search && search.trim()) {
            const likeSearch = `%${search.trim()}%`;
            baseQuery += ` AND (p.project_name LIKE ? OR p.project_id LIKE ? OR p.customer_po_value LIKE ? OR c.company_name LIKE ? OR u.name LIKE ?)`;
            params.push(likeSearch, likeSearch, likeSearch, likeSearch, likeSearch);
        }

        // Get total count
        const countQuery = baseQuery.replace(/^SELECT p\.\*, c\.company_name, u\.name as manager_name/, 'SELECT COUNT(*) as total');
        const countResult = await db.query(countQuery, params);
        const total = countResult[0]?.total || 0;

        // Add ORDER BY, LIMIT, OFFSET
        baseQuery += ` ORDER BY p.${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
        params.push(limitNum, offset);

        const projects = await db.query(baseQuery, params);
        return { projects, total };
    }

    static async datatableProjectReports({ page = 1, limit = 10, sortBy = 'created_at', sortOrder = 'DESC', search = '', userIds = [], branch = [], startdate, enddate, status }) {
        const allowedSortFields = ['created_at', 'project_name', 'project_id', 'status'];
        const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'created_at';
        const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
        const pageNum = parseInt(page);
        const limitNum = parseInt(limit);
        const offset = (pageNum - 1) * limitNum;

        let baseQuery = `SELECT p.*, u.name AS username, u.emp_id, c.company_name, b.branch_name FROM projects p LEFT JOIN users u ON p.project_manager = u.id LEFT JOIN companies c ON p.customer_id = c.id LEFT JOIN branches b ON u.branch_id = b.id WHERE 1=1`;
        let params = [];

        // Date filter
        if (startdate && enddate) {
            baseQuery += ` AND p.created_at BETWEEN ? AND ?`;
            params.push(`${startdate} 00:00:00`, `${enddate} 23:59:59`);
        }

        // User filter
        if (userIds && userIds.length > 0) {
            baseQuery += ` AND u.id IN (${userIds.map(() => '?').join(',')})`;
            params.push(...userIds);
        }

        // Branch filter
        if (branch && branch.length > 0) {
            baseQuery += ` AND u.branch_id IN (${branch.map(() => '?').join(',')})`;
            params.push(...branch);
        }

        // Status filter
        if (status) {
            baseQuery += ` AND p.status = ?`;
            params.push(status);
        }

        if (search && search.trim()) {
            const likeSearch = `%${search.trim()}%`;
            baseQuery += ` AND (p.project_name LIKE ? OR p.project_id LIKE ? OR c.company_name LIKE ? OR b.branch_name LIKE ? OR u.name LIKE ?)`;
            params.push(likeSearch, likeSearch, likeSearch, likeSearch, likeSearch);
        }

        // Get total count
        let countBaseQuery = baseQuery;
        let countQuery = `SELECT COUNT(*) as total FROM (${countBaseQuery} GROUP BY p.id) as sub`;
        const countResult = await db.query(countQuery, params);
        const total = countResult[0]?.total || 0;

        // Add ORDER BY, LIMIT, OFFSET
        baseQuery += ` GROUP BY p.id ORDER BY p.${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
        params.push(limitNum, offset);

        const rows = await db.query(baseQuery, params);
        // Format response
        const data = (rows || []).map((project, i) => ({
            sno: i + 1 + offset,
            id: project.id,
            project_id: project.project_id,
            company_name: project.company_name,
            branch_name: project.branch_name,
            title: project.project_name,
            start_date: project.start_date,
            address: project.address,
            notes: project.notes,
            payment_terms: project.payment_terms,
            ho_service_date: project.ho_service_date,
            closure_date: project.closure_date,
            closed_date: project.closed_date,
            employee_id: project.emp_id + '-' + project.username,
            status: project.status,
        }));
        return { data, total };
    }
}

export async function findAllProjectReports({ branch, userIds, startdate, enddate }) {
  // Build base query
  let baseQuery = `
    SELECT
      p.*,
      u.name AS username,
      u.emp_id,
      c.company_name,
      b.branch_name
    FROM projects p
    LEFT JOIN users u ON p.project_manager = u.id
    LEFT JOIN companies c ON p.customer_id = c.id
    LEFT JOIN branches b ON u.branch_id = b.id
    WHERE 1=1
  `;

  const params = [];

  // Date filter
  if (startdate && enddate) {
    baseQuery += ` AND p.created_at BETWEEN ? AND ?`;
    params.push(`${startdate} 00:00:00`, `${enddate} 23:59:59`);
  }

  // User filter
  if (userIds && userIds.length > 0) {
    baseQuery += ` AND u.id IN (${userIds.map(() => '?').join(',')})`;
    params.push(...userIds);
  }

  // Branch filter
  if (branch && branch.length > 0) {
    baseQuery += ` AND u.branch_id IN (${branch.map(() => '?').join(',')})`;
    params.push(...branch);
  }

  // Status filter
  /*if (status) {
    baseQuery += ` AND p.status = ?`;
    params.push(status);
  }*/

  baseQuery += ` ORDER BY p.created_at DESC`;

  const rows = await db.query(baseQuery, params);

  // Format response
  const data = (rows || []).map((project, i) => ({
    sno: i + 1,
    project_id: project.project_id,
    company_name: project.company_name,
    branch_name: project.branch_name,
    title: project.project_name,
    start_date: project.start_date,
    address: project.address,
    notes: project.notes,
    payment_terms: project.payment_terms,
    ho_service_date: project.ho_service_date,
    closure_date: project.closure_date,
    closed_date: project.closed_date,
    employee_id: project.emp_id + '-' + project.username,
    status: project.status,
    // ...add more fields as needed
  }));

  return data;
}