import { db } from '../utils/database.js';
import admin from 'firebase-admin'; // Make sure firebase-admin is initialized elsewhere

export class ProjectAssignee {
    static async create(assigneeData, userId) {
        try {
            // Start transaction
            //await db.query('START TRANSACTION');

            const insertedIds = [];
            
            // Insert multiple assignees
            for (const assigneeId of assigneeData.user_id) {
                // Check if assignee already exists
                const exists = await this.checkAssigneeExists(
                    assigneeData.project_id, 
                    assigneeId
                );
                
                if (!exists) {
                    const query = `
                        INSERT INTO projectassignees (
                            project_id, user_id, assigned_on,
                            assigned_by, status, created_at, updated_at
                        ) VALUES (?, ?, NOW(), ?, 1, NOW(), NOW())
                    `;
                    
                    const result = await db.query(query, [
                        assigneeData.project_id,
                        assigneeId,
                        userId
                    ]);
                    
                    insertedIds.push(result.insertId);

                    // Fetch device_token and project code
                    let userRow = await db.query(
                        'SELECT token, name FROM users WHERE id = ? LIMIT 1',
                        [assigneeId]
                    );
                    let projectRow = await db.query(
                        'SELECT project_id, project_name FROM projects WHERE id = ? LIMIT 1',
                        [assigneeData.project_id]
                    );
                    const deviceToken = userRow ? userRow.token : null;
                    const projectCode = projectRow ? projectRow.project_id : '';
                    const projectName = projectRow ? projectRow.project_name : '';

                    // Compose notification
                    const notificationTitle = 'Project Assigned';
                    const notificationBody = `You have been assigned to project ${projectCode} (${projectName})`;

                    // Send Firebase push notification if deviceToken exists
                    if (deviceToken) {
                        const message = {
                            notification: {
                                title: notificationTitle,
                                body: notificationBody
                            },
                            token: deviceToken
                        };
                        try {
                            await admin.messaging().send(message);
                        } catch (err) {
                            console.error('Firebase push notification error:', err);
                        }
                    }

                    // Insert into firebaseinboxes table
                    await db.query(
                        `INSERT INTO firebaseinboxes 
                            (user_id, category, message, action, trigger_date, trigger_time, read_status, project_id, task_id, created_at, updated_at)
                         VALUES (?, ?, ?, ?, ?, NOW(), ?, ?, ?, NOW(), NOW())`,
                        [
                            assigneeId,
                            'Project Assignment',
                            notificationBody,
                            'assigned',
                            new Date().toISOString().slice(0, 10), // trigger_date (YYYY-MM-DD)
                             0, // unread
                            assigneeData.project_id,
                            null // task_id
                        ]
                    );
                }
            }

            //await db.query('COMMIT');
            return insertedIds;

        } catch (error) {
            //await db.query('ROLLBACK');
            throw error;
        }
    }

    static async update(id, assigneeData) {
        try {
            // Start transaction
            await db.query('START TRANSACTION');

            // Delete existing assignees for this project
            await db.query(
                'DELETE FROM projectassignees WHERE project_id = ?', 
                [assigneeData.project_id]
            );

            // Insert new assignees
            const insertedIds = [];
            for (const assigneeId of assigneeData.user_id) {
                const query = `
                    INSERT INTO projectassignees (
                        project_id, user_id, assigned_on,
                        assigned_by, status, created_at, updated_at
                    ) VALUES (?, ?, NOW(), ?, 1, NOW(), NOW())
                `;
                
                const result = await db.query(query, [
                    assigneeData.project_id,
                    assigneeId,
                    assigneeData.assigned_by
                ]);
                
                insertedIds.push(result.insertId);
            }

            await db.query('COMMIT');
            return insertedIds;

        } catch (error) {
            await db.query('ROLLBACK');
            throw error;
        }
    }

    static async softDelete(id) {
        const query = `
            UPDATE projectassignees 
            SET status = 0,
                updated_at = NOW() 
            WHERE id = ?
        `;
        return db.query(query, [id]);
    }

    static async findById(id) {
        const query = `
            SELECT pa.*,
                   p.project_id as project_code,
                   p.customer_po,
                   u.name as assignee_name,
                   a.name as assigned_by_name
            FROM projectassignees pa
            LEFT JOIN projects p ON pa.project_id = p.id
            LEFT JOIN users u ON pa.user_id = u.id
            LEFT JOIN users a ON pa.assigned_by = a.id
            WHERE pa.id = ? AND pa.status = 1
        `;
        const [assignee] = await db.query(query, [id]);
        return assignee;
    }

    static async list(projectId) {
        const query = `
            SELECT pa.*,
                   p.project_id as project_code,
                   p.customer_po,
                   u.name as assignee_name,
                   a.name as assigned_by_name
            FROM projectassignees pa
            LEFT JOIN projects p ON pa.project_id = p.id
            LEFT JOIN users u ON pa.user_id = u.id
            LEFT JOIN users a ON pa.assigned_by = a.id
            WHERE pa.status = 1 
            AND pa.project_id = ?
            ORDER BY pa.created_at DESC
        `;
        return db.query(query, [projectId]);
    }

    static async checkAssigneeExists(projectId, userId) {
        try {
            const query = `
                SELECT EXISTS(
                    SELECT 1 
                    FROM projectassignees 
                    WHERE project_id = ? 
                    AND user_id = ?
                    AND status = 1
                ) as exists_flag
            `;
            const [[{exists_flag}]] = await db.query(query, [projectId, userId]);
            return exists_flag === 1;
        } catch (error) {
            console.error('Check assignee exists error:', error);
            return false;
        }
    }
}