import { db } from "../utils/database.js";

function toNull(val) {
    return val === undefined ? null : val;
}

export async function findProjectsRolewise({ scope, userId, branchId, startDate, endDate }) {
    try {
        let query = `
            SELECT p.*, 
                   c.company_name as customer_name,
                   u.name as manager_name,
                   pk.package_name
            FROM projects p
            LEFT JOIN companies c ON p.customer_id = c.id
            LEFT JOIN users u ON p.project_manager = u.id
            LEFT JOIN packages pk ON p.package = pk.id
            WHERE p.status = 1
        `;
        let params = [];

        // Role-based filtering
        if (scope === 'owned') {
            query += `
                AND EXISTS (
                    SELECT 1 FROM projectassignees pa
                    WHERE pa.project_id = p.id AND pa.user_id = ?
                )
            `;
            params.push(toNull(userId));
        } else if (scope === 'branch') {
            // branch_id can be comma separated, handle as array
            const branchIds = (branchId || '').toString().split(',').map(id => id.trim()).filter(id => id.length > 0);
            if (branchIds.length > 0) {
                query += ` AND p.customer_branch_id IN (${branchIds.map(() => '?').join(',')}) `;
                params.push(...branchIds);
            }
        }
        // If global, no extra filter

        // Date filter
        if (startDate && endDate) {
            query += ` AND DATE(p.created_at) BETWEEN ? AND ? `;
            params.push(startDate, endDate);
        } else if (startDate) {
            query += ` AND DATE(p.created_at) = ? `;
            params.push(startDate);
        } else if (endDate) {
            query += ` AND DATE(p.created_at) = ? `;
            params.push(endDate);
        }

        query += ` ORDER BY p.created_at DESC`;

        let projects = await db.query(query, params);

        // Fetch all packages for each project
        for (let project of projects) {
            const packageRows = await db.query(
                `SELECT p.package_name FROM projectpackages pp join packages p on(p.id=pp.package_id) WHERE pp.project_id = ?`,
                [project.id]
            );
            project.project_packages = packageRows.map(row => row.package_name);
        }

        return projects;
    } catch (error) {
        console.error("Error fetching projects rolewise:", error);
        throw error;
    }
}