import { db } from '../utils/database.js';

export class Role {
    static async create(roleData) {
        // Convert eligible_transport array to comma-separated string
        const eligibleTransport = Array.isArray(roleData.eligible_transport) 
            ? roleData.eligible_transport.join(',') 
            : roleData.eligible_transport;

        const query = `
            INSERT INTO roles (
                name, scope, reporting_to, location, 
                eligible_transport, default_transport, access_scope, 
                active, created_at, updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, 1, NOW(), NOW())
        `;
        const result = await db.query(query, [
            roleData.name,
            roleData.scope,
            roleData.reporting_to,
            roleData.location,
            eligibleTransport,
            roleData.default_transport,
            roleData.access_scope
        ]);
        return result.insertId;
    }

    static async update(id, roleData) {
        // Convert eligible_transport array to comma-separated string for update
        const eligibleTransport = Array.isArray(roleData.eligible_transport) 
            ? roleData.eligible_transport.join(',') 
            : roleData.eligible_transport;

        const query = `
            UPDATE roles 
            SET name = ?, 
                scope = ?,
                reporting_to = ?,
                location = ?,
                eligible_transport = ?,
                default_transport = ?,
                access_scope = ?,
                updated_at = NOW()
            WHERE id = ? AND active = 1
        `;
        return db.query(query, [
            roleData.name,
            roleData.scope,
            roleData.reporting_to,
            roleData.location,
            eligibleTransport,
            roleData.default_transport,
            roleData.access_scope,
            id
        ]);
    }

    static async softDelete(id) {
        const query = 'UPDATE roles SET active = 0, updated_at = NOW() WHERE id = ?';
        return db.query(query, [id]);
    }

    static async findById(id) {
        const query = 'SELECT * FROM roles WHERE id = ? AND active = 1';
        const [role] = await db.query(query, [id]);
        
        if (role) {
            // Convert comma-separated string back to array
            role.eligible_transport = role.eligible_transport ? 
                role.eligible_transport.split(',') : [];
        }
        
        return role;
    }

    static async list() {
        const query = `
            SELECT r.*, rt.name AS reporting_to_role_name
            FROM roles r
            LEFT JOIN roles rt ON r.reporting_to = rt.id
            WHERE r.active = 1
            ORDER BY r.created_at DESC
        `;
        return db.query(query);
    }
}