import { db } from "../utils/database.js";

function toNull(val) {
    return val === undefined ? null : val;
}

async function findTasksRolewise({ scope, userId, branchId, startDate, endDate, projectId, tasksubcategoryId }) {
    try {
        let query = `
            SELECT 
                t.*, 
                u.emp_id, 
                u.name, 
                tt.activity_name AS tasktype_name,
                p.project_id AS project_code,
                p.project_name AS project_name,
                c.company_name AS customer_name,
                ct.contact_name AS contact_name
            FROM tasks t
            LEFT JOIN users u ON t.user_id = u.id
            LEFT JOIN roles r ON u.roles_id = r.id
            LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
            LEFT JOIN projects p ON t.project_id = p.id
            LEFT JOIN companies c ON p.customer_id = c.id
            LEFT JOIN contacts ct ON p.contact_id = ct.id
            WHERE t.tasktype_id != 1
        `;
        let params = [];
        console.log("Scope:", scope);
        // Date filtering
        if (startDate && endDate) {
            query += " AND DATE(t.start_date) BETWEEN ? AND ?";
            params.push(toNull(startDate), toNull(endDate));
        } else if (startDate) {
            query += " AND DATE(t.start_date) >= ?";
            params.push(toNull(startDate));
        } else if (endDate) {
            query += " AND DATE(t.start_date) <= ?";
            params.push(toNull(endDate));
        }

        // Project filtering
        if (projectId) {
            query += " AND t.project_id = ?";
            params.push(toNull(projectId));
        }

        // Tasksubcategory filtering
        if (tasksubcategoryId) {
            query += " AND t.tasksubcategory_id = ?";
            params.push(toNull(tasksubcategoryId));
        }

        // Role-based filtering
        if (scope === 'Global') {
            // No extra filter, all records
        } else if (scope === 'Branch') {
            query += " AND u.branch_id = ?";
            params.push(toNull(branchId));
        } else {
            query += " AND t.user_id = ?";
            params.push(toNull(userId));
        }

        // Final check: replace any undefined in params with null
        params = params.map(p => p === undefined ? null : p);

        let tasks = await db.query(query, params);
        return tasks;
    } catch (error) {
        console.error("Error fetching tasks rolewise:", error);
        throw error;
    }
}


async function finddatatableTasksRolewise({
    page,
    limit,
    sortBy,
    sortOrder,
    search,
    start_date,
    end_date,
    project_id,
    tasksubcategory_id,
    scope,
    userId,
    branchId
}) {
    try {
        const offset = (page - 1) * limit;

        // Validate and default sortBy and sortOrder
        const allowedSortFields = [
            't.created_at', 'u.name', 'p.project_name', 'tt.activity_name', 't.start_date'
        ];
        const safeSortBy = allowedSortFields.includes(sortBy) ? sortBy : 't.created_at';
        const safeSortOrder = ['ASC', 'DESC'].includes(sortOrder?.toUpperCase()) ? sortOrder.toUpperCase() : 'DESC';

        let whereClause = `WHERE t.tasktype_id != 1`;
        let params = [];

        // Search filter
        if (search) {
            whereClause += ` AND (
                u.name LIKE ? OR
                p.project_name LIKE ? OR
                tt.activity_name LIKE ? OR
                c.company_name LIKE ? OR
                ct.contact_name LIKE ?
            )`;
            const searchTerm = `%${search}%`;
            params.push(searchTerm, searchTerm, searchTerm, searchTerm, searchTerm);
        }

        // Date filter
        if (start_date && end_date) {
            whereClause += ` AND DATE(t.start_date) BETWEEN ? AND ?`;
            params.push(toNull(start_date), toNull(end_date));
        } else if (start_date) {
            whereClause += ` AND DATE(t.start_date) >= ?`;
            params.push(toNull(start_date));
        } else if (end_date) {
            whereClause += ` AND DATE(t.start_date) <= ?`;
            params.push(toNull(end_date));
        }

        // Project filter
        if (project_id !== undefined) {
            whereClause += ` AND t.project_id = ?`;
            params.push(toNull(project_id));
        }

        // Tasksubcategory filter
        if (tasksubcategory_id !== undefined) {
            whereClause += ` AND t.tasksubcategory_id = ?`;
            params.push(toNull(tasksubcategory_id));
        }

        // Role-based filtering
        if (scope === 'Branch' && branchId !== undefined) {
            whereClause += ` AND u.branch_id = ?`;
            params.push(toNull(branchId));
        } else if (scope !== 'Global' && userId !== undefined) {
            whereClause += ` AND t.user_id = ?`;
            params.push(toNull(userId));
        }

        const baseQuery = `
            FROM tasks t
            LEFT JOIN users u ON t.user_id = u.id
            LEFT JOIN roles r ON u.roles_id = r.id
            LEFT JOIN tasktypes tt ON t.tasktype_id = tt.id
            LEFT JOIN projects p ON t.project_id = p.id
            LEFT JOIN companies c ON p.customer_id = c.id
            LEFT JOIN contacts ct ON p.contact_id = ct.id
        `;

        const dataQuery = `
            SELECT
                t.*,
                u.emp_id,
                u.name,
                tt.activity_name AS tasktype_name,
                p.project_id AS project_code,
                p.project_name,
                c.company_name AS customer_name,
                ct.contact_name AS contact_name
            ${baseQuery}
            ${whereClause}
            ORDER BY ${safeSortBy} ${safeSortOrder}
            LIMIT ? OFFSET ?
        `;

        const countQuery = `
            SELECT COUNT(*) AS total
            ${baseQuery}
            ${whereClause}
        `;

        const dataParams = [...params, limit, offset];
        const safeDataParams = dataParams.map(toNull);
        const safeCountParams = params.map(toNull);

        const [tasks, countRows] = await Promise.all([
            db.query(dataQuery, safeDataParams),
            db.query(countQuery, safeCountParams)
        ]);

        const total = countRows[0]?.total || 0;

        return { tasks, total };
    } catch (error) {
        console.error("Error in finddatatableTasksRolewise:", error);
        throw error;
    }
}

export {
    findTasksRolewise,    
    finddatatableTasksRolewise
};