import { db } from '../utils/database.js';

export class TaskSubcategory {
    static async create(categoryData) {
        try {
            //await db.query('START TRANSACTION');

            // Insert subcategory
            const query = `
                INSERT INTO tasksubcategories (
                    tasktype_id, subcategory, show_productdropdown, 
                    status, created_at, updated_at
                ) VALUES (?, ?,?, 1, NOW(), NOW())
            `;
            
            const result = await db.query(query, [
                categoryData.tasktype_id,
                categoryData.subcategory,
                categoryData.show_productdropdown || 0
            ]);
            
            const subcategoryId = result.insertId;

            // Insert options if provided
            if (categoryData.options && categoryData.options.length > 0) {
                const optionsQuery = `
                    INSERT INTO taskcategoryoptions (
                        tasktype_id, tasksubcategory_id, dropdown,
                        status, created_at, updated_at
                    ) VALUES (?, ?, ?, 1, NOW(), NOW())
                `;

                for (const option of categoryData.options) {
                    await db.query(optionsQuery, [
                        categoryData.tasktype_id,
                        subcategoryId,
                        option.dropdown
                    ]);
                }
            }
             // Insert action items if provided
        if (categoryData.actions && categoryData.actions.length > 0) {
            const actionQuery = `
                INSERT INTO taskactionlists (
                    task_type_id, tasksubcategory_id, action_item,
                    shortname, status, create_date, created_at, updated_at
                ) VALUES (?, ?, ?, ?, 1, NOW(), NOW(), NOW())
            `;

            for (const action of categoryData.actions) {
                await db.query(actionQuery, [
                    categoryData.tasktype_id,
                    subcategoryId,
                    action.action_item,
                    action.shortname
                ]);
            }
        }
            //await db.query('COMMIT');
            return subcategoryId;

        } catch (error) {
            //await db.query('ROLLBACK');
            throw error;
        }
    }

    static async update(id, categoryData) {
        try {
           // await db.query('START TRANSACTION');

            // Update subcategory
            const query = `
                UPDATE tasksubcategories 
                SET tasktype_id = ?,
                    subcategory = ?,
                    show_productdropdown = ?,
                    updated_at = NOW()
                WHERE id = ? AND status = 1
            `;
            
            await db.query(query, [
                categoryData.tasktype_id,
                categoryData.subcategory,
                categoryData.show_productdropdown || 0,
                id
            ]);

            // Delete existing options
            await db.query(
                'DELETE FROM taskcategoryoptions WHERE tasksubcategory_id = ?',
                [id]
            );

            // Insert new options if provided
            if (categoryData.options && categoryData.options.length > 0) {
                const optionsQuery = `
                    INSERT INTO taskcategoryoptions (
                        tasktype_id, tasksubcategory_id, 
                         dropdown,
                        status, created_at, updated_at
                    ) VALUES (?, ?, ?, 1, NOW(), NOW())
                `;

                for (const option of categoryData.options) {
                    await db.query(optionsQuery, [
                        categoryData.tasktype_id,
                        id,
                        option.dropdown
                    ]);
                }
            }
            // Delete existing actions
        await db.query(
            'DELETE FROM taskactionlists WHERE tasksubcategory_id = ?',
            [id]
        );

        // Insert new action items if provided
        if (categoryData.actions && categoryData.actions.length > 0) {
            const actionQuery = `
                INSERT INTO taskactionlists (
                    task_type_id, tasksubcategory_id, action_item,
                    shortname, status, create_date, created_at, updated_at
                ) VALUES (?, ?, ?, ?, 1, NOW(), NOW(), NOW())
            `;

            for (const action of categoryData.actions) {
                await db.query(actionQuery, [
                    categoryData.tasktype_id,
                    id,
                    action.action_item,
                    action.shortname
                ]);
            }
        }

            //await db.query('COMMIT');
            return id;

        } catch (error) {
            //await db.query('ROLLBACK');
            throw error;
        }
    }

    static async softDelete(id) {
        const query = `
            UPDATE tasksubcategories 
            SET status = 0,
                updated_at = NOW() 
            WHERE id = ?
        `;
        return db.query(query, [id]);
    }

    static async findById(id) {
        // Get subcategory
        const categoryQuery = `
            SELECT ts.*, tt.activity_name
            FROM tasksubcategories ts
            LEFT JOIN tasktypes tt ON ts.tasktype_id = tt.id
            WHERE ts.id = ? AND ts.status = 1
        `;
        const [category] = await db.query(categoryQuery, [id]);

        if (category) {
            // Get options (removed product join)
            const optionsQuery = `
                SELECT tco.*
                FROM taskcategoryoptions tco
                WHERE tco.tasksubcategory_id = ? 
                AND tco.status = 1
            `;
            category.options = await db.query(optionsQuery, [id]);
            // Get task action lists for this task subcategory
            const actionListsQuery = `
                SELECT tal.*
                FROM taskactionlists tal
                WHERE tal.tasksubcategory_id = ?
                AND tal.status = 1
                ORDER BY tal.id ASC
            `;
            category.action_lists = await db.query(actionListsQuery, [id]);
        }
        return category;
    }
    static async list() {
        // Get all subcategories
        const categoryQuery = `
            SELECT ts.*, tt.activity_name
            FROM tasksubcategories ts
            LEFT JOIN tasktypes tt ON ts.tasktype_id = tt.id
            WHERE ts.status = 1
            ORDER BY ts.created_at DESC
        `;
        const categories = await db.query(categoryQuery);

        // Get options for each category (without product join)
        for (const category of categories) {
            const optionsQuery = `
                SELECT tco.*
                FROM taskcategoryoptions tco
                WHERE tco.tasksubcategory_id = ? 
                AND tco.status = 1
            `;
            category.options = await db.query(optionsQuery, [category.id]);
        }

        return categories;
    }

    static async getByTaskType(taskTypeId) {
        try {
            const query = `
                SELECT 
                    ts.id, ts.subcategory as subcategory, ts.tasktype_id, ts.show_productdropdown,
                    tco.id as option_id, tco.dropdown as option_name, ts.show_suboption
                FROM tasksubcategories ts
                LEFT JOIN taskcategoryoptions tco ON ts.id = tco.tasksubcategory_id
                WHERE ts.tasktype_id = ?
                AND ts.status = 1
            `;

            let results = await db.query(query, [taskTypeId]);
            const resultsArray = !Array.isArray(results) ? [results] : results;

            const subcategories = {};
            resultsArray.forEach(row => {
                if (!row || !row.id) return;

                if (!subcategories[row.id]) {
                    subcategories[row.id] = {
                        id: row.id,
                        name: row.subcategory,
                        tasktype_id: row.tasktype_id,
                        show_productdropdown: row.show_productdropdown === 1,
                        show_suboption: row.show_suboption === 1,
                        options: []
                    };
                }

                if (row.option_id) {
                    const option = {
                        id: row.option_id,
                        name: row.option_name
                    };
                    if (!subcategories[row.id].options.find(o => o.id === option.id)) {
                        subcategories[row.id].options.push(option);
                    }
                }
            });

            return Object.values(subcategories);
        } catch (error) {
            console.error('Get task subcategories error:', error);
            throw error;
        }
    }

}