import { db } from '../utils/database.js';
import { hashPassword } from '../utils/hash.js';

export class User {
    static async checkMobileExists(mobile) {
        try {
            const query = `
                SELECT COUNT(*) as count 
                FROM users 
                WHERE mobile = ? 
                AND active = 1
            `;
            
            // Debug log
            console.log('Checking mobile:', mobile);

            // Get query results with proper destructuring
            let results = await db.query(query, [mobile]);
            
            // Debug log
            console.log('Query results:', results);

            // Check if we have results and extract count
            if (!results || !results[0]) {
                return false;
            }

            return results[0].count > 0;
        } catch (error) {
            console.error('Check mobile exists error:', error);
            throw error;
        }
    }

    static async create(userData, createdBy) {
        // Check if mobile number already exists
        const mobileExists = await this.checkMobileExists(userData.mobile);
        if (mobileExists) {
            throw new Error('Mobile number already exists');
        }

        // Convert branch_id array to comma-separated string
        const branchIds = Array.isArray(userData.branch_id) 
            ? userData.branch_id.join(',') 
            : userData.branch_id;

        const query = `
            INSERT INTO users (
                branch_id, emp_id, name, email, password, mobile,
                reporting_to, join_date, profile_pic, work_number,
                roles_id,ip_address, create_by, active, create_date,
                created_at, updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1, NOW(), NOW(), NOW())
        `;
        
        const result = await db.query(query, [
            branchIds,
            userData.emp_id,
            userData.name,
            userData.email,
            userData.password,
            userData.mobile,
            userData.reporting_to || null,
            userData.join_date,
            userData.profile_pic || null,
            userData.work_number,
            userData.roles_id,
            userData.ip,
            createdBy
        ]);
        
        const date = new Date().toISOString().split('T')[0];
        console.log(`ID: ${result.insertId}, Name: ${userData.name}`);
           // console.log(user.id, user.name);
            const taskTitle = `Attendance ${userData.name}-${date}`;
            const description = taskTitle;

           let result1 = await db.query(
                `INSERT INTO tasks 
                (task, description, start_date, start_time, end_date, end_time, user_id, status,created_date, created_at, updated_at,tasktype_id)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?,NOW(), NOW(), NOW(),1)`,
                [
                    taskTitle,
                    description,
                    date,
                    '09:00 AM',
                    date,
                    '06:00 PM',
                    result.insertId,
                    'Open'
                ]
            );
        return result.insertId;
    }

    /*static async update(id, userData) {
        // Convert branch_id array to comma-separated string for update
        const branchIds = Array.isArray(userData.branch_id) 
            ? userData.branch_id.join(',') 
            : userData.branch_id;

        const query = `
            UPDATE users 
            SET branch_id = ?,
                emp_id = ?,
                name = ?,
                email = ?,
                mobile = ?,
                reporting_to = ?,
                join_date = ?,
                profile_pic = ?,
                work_number = ?,
                roles_id = ?,
                ip_address = ?,
                updated_at = NOW()
            WHERE id = ? AND active = 1
        `;
        
        return db.query(query, [
            branchIds,
            userData.emp_id,
            userData.name,
            userData.email,
            userData.mobile,
            userData.reporting_to || null,
            userData.join_date,
            userData.profile_pic || null,
            userData.work_number,
            userData.roles_id,
            userData.ip,
            id
        ]);
    }*/
    
static async update(id, userData) {
    const branchIds = Array.isArray(userData.branch_id)
        ? userData.branch_id.join(',')
        : userData.branch_id;

    // Base query and parameters
    let query = `
        UPDATE users 
        SET branch_id = ?,
            emp_id = ?,
            name = ?,
            email = ?,
            mobile = ?,
            reporting_to = ?,
            join_date = ?,
            profile_pic = ?,
            work_number = ?,
            roles_id = ?,
            ip_address = ?,
    `;

    const params = [
        branchIds,
        userData.emp_id,
        userData.name,
        userData.email,
        userData.mobile,
        userData.reporting_to || null,
        userData.join_date,
        userData.profile_pic || null,
        userData.work_number,
        userData.roles_id,
        userData.ip,
    ];

    // Conditionally add password
    if (userData.password) {
        const hashedPassword = await hashPassword(userData.password);
        query += `password = ?,`;
        params.push(hashedPassword);
    }

    // Finalize query
    query += `updated_at = NOW() WHERE id = ? AND active = 1`;
    params.push(id);

    return db.query(query, params);
}
    static async findById(id) {
        const query = `
            SELECT u.*, r.name as role_name 
            FROM users u
            LEFT JOIN roles r ON u.roles_id = r.id
            WHERE u.id = ? AND u.active = 1
        `;
        const [user] = await db.query(query, [id]);
        
        if (user) {
            // Convert comma-separated branch_ids back to array
            user.branch_id = user.branch_id ? user.branch_id.split(',').map(Number) : [];
            
            // Get branch details
            if (user.branch_id.length > 0) {
                const branchQuery = `
                    SELECT id, branch_name 
                    FROM branches 
                    WHERE FIND_IN_SET(id, ?)
                `;
                const branches = await db.query(branchQuery, [user.branch_id.join(',')]);
                user.branches = branches;
            } else {
                user.branches = [];
            }
        }
        
        return user;
    }

    static async list() {
        const query = `
            SELECT u.*, 
                   r.name as role_name,
                   ru.name as reporting_to_user_name
            FROM users u
            LEFT JOIN roles r ON u.roles_id = r.id
            LEFT JOIN users ru ON u.reporting_to = ru.id
            WHERE u.active = 1 
            ORDER BY u.created_at DESC
        `;
        const users = await db.query(query);

        // Convert comma-separated branch_ids back to array and get branch details
        for (let user of users) {
            user.branch_id = user.branch_id ? user.branch_id.split(',').map(Number) : [];
            if (user.branch_id.length > 0) {
                const branchQuery = `
                    SELECT id, branch_name 
                    FROM branches 
                    WHERE FIND_IN_SET(id, ?)
                `;
                const branches = await db.query(branchQuery, [user.branch_id.join(',')]);
                user.branches = branches;
            } else {
                user.branches = [];
            }

        // Fetch latest applog entry for the user
        const applogQuery = `
            SELECT app_version 
            FROM applogs 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT 1
        `;
        const [applog] = await db.query(applogQuery, [user.id]);
        user.app_logid = applog?.app_version ?? '';
        }

        return users;
    }

    static async datatableList({ page = 1, limit = 10, sortBy = 'created_at', sortOrder = 'DESC', search = '' }) {
        const allowedSortFields = ['created_at', 'name', 'mobile', 'email'];
        const validSortBy = allowedSortFields.includes(sortBy) ? sortBy : 'created_at';
        const validSortOrder = sortOrder && sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
        const pageNum = parseInt(page);
        const limitNum = parseInt(limit);
        const offset = (pageNum - 1) * limitNum;

        let baseQuery = `SELECT u.*, ru.name as reporting_to_user_name FROM users u LEFT JOIN users ru ON u.reporting_to = ru.id`;
        let params = [];
        console.log("test"+search);
        /*if (search!='' && search.trim()) {
            const likeSearch = `%${search.trim()}%`;
            baseQuery += ` WHERE (u.name LIKE ? OR u.mobile LIKE ? OR u.email LIKE ?)`;
            params.push(likeSearch, likeSearch, likeSearch);
        }*/
        const likeSearch = `%${search.trim()}%`;
        if (search !== '' && search.trim()) {
         baseQuery += ` WHERE u.active = 1 AND (u.name LIKE ? OR u.mobile LIKE ? OR u.email LIKE ?)`;
         params.push(likeSearch, likeSearch, likeSearch);
        } else {
          baseQuery += ` WHERE u.active = 1`;
        }
        // Get total count
        const countQuery = baseQuery.replace(/^SELECT u\.\*, ru\.name as reporting_to_user_name/, 'SELECT COUNT(*) as total');
        const countResult = await db.query(countQuery, params);
        const total = countResult[0]?.total || 0;

        // Add ORDER BY, LIMIT, OFFSET
        baseQuery += ` ORDER BY u.${validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
        params.push(limitNum, offset);
        console.log(baseQuery);
        const users = await db.query(baseQuery, params);
        // Convert comma-separated branch_ids back to array and get branch details
        for (let user of users) {
            user.branch_id = user.branch_id ? user.branch_id.split(',').map(Number) : [];
            if (user.branch_id.length > 0) {
                const branchQuery = `SELECT id, branch_name FROM branches WHERE FIND_IN_SET(id, ?)`;
                const branches = await db.query(branchQuery, [user.branch_id.join(',')]);
                user.branches = branches;
            } else {
                user.branches = [];
            }

        // Fetch latest applog entry for the user
        const applogQuery = `
            SELECT app_version 
            FROM applogs 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT 1
        `;
        const [applog] = await db.query(applogQuery, [user.id]);
        user.app_logid = applog?.app_version ?? '';
        }

        return { users, total };
    }

    static async findByMobile(mobile) {
        console.log('Finding user by mobile:', mobile);
        if (!mobile) {
            throw new Error('Mobile number is required');
        }
        const query = `
            SELECT u.*, r.name as role_name
            FROM users u
            LEFT JOIN roles r ON u.roles_id = r.id
            WHERE u.mobile = ? AND u.active = 1
        `;
        //const query = 'SELECT * FROM users WHERE mobile = ?';
        const [user] = await db.query(query, [mobile]);
        console.log('User found:', user);
        if (!user || user.length === 0) {
            console.log('No user found with the provided mobile number');
            return null;
        }
        return user;
    }

    static async getUserBranches(userId) {
        const query = `
            SELECT id, branch_name 
            FROM branches 
            WHERE FIND_IN_SET(id, (
                SELECT branch_id 
                FROM users 
                WHERE id = ?
            ))
        `;
        const branches = await db.query(query, [userId]);
        return branches;
    }

    static async softDelete(id) {
        const query = `
            UPDATE users 
            SET active = 0, 
                deactive_date = NOW(),
                updated_at = NOW() 
            WHERE id = ?
        `;
        return db.query(query, [id]);
    }

    static async getUsersByRole(roleId) {
        try {
            // Get the reporting_to role for the given roleId
            const [role] = await db.query('SELECT reporting_to FROM roles WHERE id = ?', [roleId]);
            if (!role || !role.reporting_to) {
                return [];
            }
            const reportingRoleId = role.reporting_to;

            // Get users with the reporting_to role
            const userQuery = `
                SELECT u.*,
                    r.name as role_name,
                    r.reporting_to as reporting_role_id,
                    r2.name as reporting_role_name
                FROM users u
                INNER JOIN roles r ON u.roles_id = r.id
                LEFT JOIN roles r2 ON r.reporting_to = r2.id
                WHERE u.active = 1 
                AND u.roles_id = ?
                ORDER BY u.created_at DESC
            `;
            const users = await db.query(userQuery, [reportingRoleId]);
            return users;
        } catch (error) {
            console.error('Get users by role error:', error);
            throw error;
        }
    }

    static async findByMobileWithRelations(mobile) {
        const query = `
            SELECT u.*,
                   r.id as role_id, r.name as rolename, r.access_scope, r.reporting_to as role_reporting_to,
                   r.location, r.default_transport, r.eligible_transport as role_transport,
                   r.active as role_status, r.device_approval as role_device_approval,
                   ur.id as reporting_user_id, 
                   ur.emp_id as reporting_user_employee_id,
                   ur.name as reporting_user_name,
                   ur.email as reporting_user_email,
                   ur.work_number as reporting_user_work_number,
                   ur.mobile as reporting_user_mobile,
                   ur.reporting_to as reporting_user_reporting_to,
                   ur.join_date as reporting_user_doj,
                   ur.create_date as reporting_user_created_date,
                   ur.create_by as reporting_user_created_by,
                   ur.profile_pic as reporting_user_profile_pic,
                   ur.branch_id as reporting_user_branch_id,
                   ur.active as reporting_user_status,
                   ur.device_token as reporting_user_device_token,
                   ur.device_id as reporting_user_device_id
            FROM users u
            LEFT JOIN roles r ON u.roles_id = r.id
            LEFT JOIN users ur ON u.reporting_to = ur.id
            WHERE u.mobile = ? AND u.active = 1
        `;
        
        const [user] = await db.query(query, [mobile]);
        
        if (user) {
            // Format user role data
            user.userrole = {
                id: user.role_id,
                role: user.rolename,
                reporting_to: user.role_reporting_to,
                location_tracking: user.location,
                vehicle: user.default_tarnsport,
                eligible_transport: user.role_transport,
                status: user.role_status,
                device_approval: user.role_device_approval
            };

            // Format reporting user data
            user.userreporting = {
                id: user.reporting_user_id,
                employee_id: user.reporting_user_employee_id,
                external_id: user.reporting_user_external_id,
                name: user.reporting_user_name,
                email: user.reporting_user_email,
                work_number: user.reporting_user_work_number,
                mobile_no: user.reporting_user_mobile,
                reporting_to: user.reporting_user_reporting_to,
                doj: user.reporting_user_doj,
                created_date: user.reporting_user_created_date,
                created_by: user.reporting_user_created_by,
                profile_pic: user.reporting_user_profile_pic,
                branch_id: user.reporting_user_branch_id,
                status: user.reporting_user_status,
                device_token: user.reporting_user_device_token,
                device_id: user.reporting_user_device_id
            };

            // Remove temporary fields
            Object.keys(user).forEach(key => {
                if (key.startsWith('role_') || key.startsWith('reporting_user_')) {
                    delete user[key];
                }
            });
        }

        return user;
    }

    static async updateDeviceInfo(userId, deviceInfo) {
        console.log("user id"+userId);
        console.log("device details: " + JSON.stringify(deviceInfo));
        try {

        //console.log("device details"+deviceInfo);
        const query = `
            UPDATE users 
            SET device_token = ?,
                device_id = ?,
                device_info = ?,
                updated_at = NOW()
            WHERE id = ?
        `;
        
        return db.query(query, [
            deviceInfo.device_token,
            deviceInfo.device_id,
            deviceInfo.app_version,
            userId
        ]);
        } catch (err) {
    console.error("DB Update Error:", err);
    throw err;
    }
}

    static async changePassword(userId, newPassword, confirmPassword) {
        // 1. Check if newPassword and confirmPassword match
        if (newPassword !== confirmPassword) {
            return { error: 'New password and confirm password do not match' };
        }

        // 2. Hash new password
        const newPasswordHash = hashPassword(newPassword);

        // 3. Update password
        await db.query('UPDATE users SET password = ?, updated_at = NOW() WHERE id = ?', [newPasswordHash, userId]);
        return { success: true };
    }

    static async updateHomeLocation(userId, home_latitude, home_longitude, home_address) {
        console.log('Updating home location for user:', userId);
        const query = `
            UPDATE users
            SET home_latitude = ?,
                home_longitude = ?,
                home_address = ?,
                home_approval = 'Pending',
                updated_at = NOW()
            WHERE id = ?
        `;
        await db.query(query, [home_latitude, home_longitude, home_address, userId]);
        return { success: true };
    }
     static async approveHomeLocation(userId, home_address) {
        console.log('Updating home location for user:', userId);
        const query = `
            UPDATE users
            SET home_approval = 'Approved',
                updated_at = NOW()
            WHERE id = ?
        `;
        await db.query(query, [userId]);
        return { success: true };
    }
    static async approveDevice(userId) {
        try {
            console.log("device update"+userId);
            // First get the user's temp_device_id
            let user = await db.query('SELECT temp_device_id FROM users WHERE id = ?', [userId]);
            console.log(`user details: ${JSON.stringify(user)}`);

            
            if (!user || !user[0] || !user[0].temp_device_id) {
                throw new Error('User not found or no temporary device ID exists', userId);
            }
            
            // Clear device_id for any users using the same device_id as the temp_device_id
            await db.query(
                'UPDATE users SET device_id = "" WHERE device_id = ?', 
                [user[0].temp_device_id]
            );
            
            // Update the user's device info
            const query = `
                UPDATE users 
                SET device_id = temp_device_id,
                    temp_device_id = '',
                    device_approval = 0,
                    updated_at = NOW()
                WHERE id = ?
            `;
            
            await db.query(query, [userId]);
            return { success: true };
        } catch (error) {
            console.error('Approve device error:', error);
            throw error;
        }
    }

    static async changeDeviceToken(mobile_no, deviceId) {
        // Find user by mobile number
        const [user] = await db.query('SELECT * FROM users WHERE mobile = ? AND active = 1', [mobile_no]);
        if (!user) {
            return null;
        }
        // Only allow for roles_id 3 or 5
        if (user.roles_id !== 3 && user.roles_id !== 4) {
            return 'invalid_role';
        }
        // Update temp_device_id, device_info, device_approval
        await db.query(
            'UPDATE users SET temp_device_id = ?, device_approval = 1, updated_at = NOW() WHERE id = ?',
            [deviceId, user.id]
        );
        // Return updated user
        let updatedUser = await db.query('SELECT * FROM users WHERE id = ?', [user.id]);
        return updatedUser;
    }
}
